/* Standalone analysis program, designed to read only integer, mono 
 * sound files.  Seems to be able to squeeze in 34 pole analysis.
 * cc -o anallpc anallpc.c -lsf -lndir -l42 -lm
 * Right now it is scaled to a maximum of 34 poles and a maximum
 * segment size of 500, which means that the frame slices must not
 * exceed 250.  On a separate i/d machine these can be comfortably
 * increased.  This program takes about 58k as scaled here. */

#include <sys/soundfile.h>
#include <sys/stat.h>
#include <stdio.h>
#include <math.h>

int NPOLE;
int FRAME;
int NPP1;

#define POLEMAX 34 
#define FRAMAX   500 
#define NDATA 4 /* number of data values stored with frame */

main()
{
	int jj,slice,ii,counter;
	float coef[POLEMAX+NDATA],inskip,dur;
	char name[36];
	double errn,rms1,rms2,cc[40];
	int sigi[FRAMAX];
	long i,nsamps,begbyte,outskip;
	struct SFDESC sfdesc;
	struct stat stat;
	short *cyllist;
	int anal,sound,nbytes,nblpc,firstframe,nread;

	fprintf(stderr,"Enter name of output data file\t");
	gets(name);
	if((anal = open(name,2)) < 0) {
		fprintf(stderr," Can't open data file\n");
		exit(1);
	}

	fprintf(stderr,"Enter name of soundfile to analyze\t");
	gets(name);
	if(sfstat(name,&sfdesc,&stat,&cyllist)) {
		fprintf(stderr," Can't stat file \n");
		exit(1); 
	}

	if(sfdesc.sf_chans != 1) {
		fprintf(stderr," Can only analyze mono file, sorry\n");
		exit(1);
	}

	if(sfdesc.sf_class != INT) {
		fprintf(stderr," Can only analyze integer file\n");
		exit(1);
	}

	if((sound = sfopen(name,READ)) < 0) {
		fprintf(stderr," Can't open sound file\n");
		exit(1);
	}

	fprintf(stderr,"Enter number of poles and frame size\t");
	scanf("%d %d",&NPOLE,&slice);
	if(NPOLE > POLEMAX) {   
		fprintf(stderr,"exceeds maximum allowed\n");
		exit(1);
	}

	FRAME = slice * 2;
	if(FRAME > FRAMAX) {
		fprintf(stderr,"exceeds maximum allowed\n");
		exit(1);
	}

	fprintf(stderr,"Specify input skip, and duration to analyze\t");
	scanf("%f %f",&inskip,&dur);

	begbyte = (long)(inskip * sfdesc.sf_srate) * (long)sfdesc.sf_class;
	printf(" begbyte %D\n",begbyte);
	if(sflseek(sound,begbyte,0) < 0) {
		fprintf(stderr,"Bad sflseek\n");
		exit(1);
	}

	fprintf(stderr,"Specify starting output frame number\t");
	scanf("%d",&firstframe);

	NPP1 = NPOLE+1;
	nblpc = (NPOLE + NDATA)*FLOAT;
	outskip = firstframe * nblpc;

	if((lseek(anal,outskip,0)) < 0) {
		fprintf(stderr,"Bad lseek on analysis file\n");
		exit(1);
	}

	nsamps = dur * sfdesc.sf_srate;
	if((nread = sfread(sound, (char *)sigi, FRAME * sfdesc.sf_class)) !=
	    FRAME * sfdesc.sf_class) {
		fprintf(stderr," Bad sfread, nread = %d\n",nread);
		exit(1);
	}
/*	for(i=0;i<FRAME;i++) printf("%d\n",sigi[i]); */


	i = 0;
	counter = 1;
	while(i < nsamps) {

		alpol(sigi,&errn,&rms1,&rms2,cc);
		coef[0] = (float)rms2;
		coef[1] = (float)rms1;
		coef[2] = (float)errn;

		fprintf(stderr,"%d %f %f %f %f\n",
			counter++,coef[0],coef[1],coef[2],coef[3]);

		coef[3] = 0.;  /*save for pitch of frame */

		for(jj=NDATA; jj<NPOLE+NDATA; jj++) /* reverse order and change sign */
			coef[jj] = (float)-cc[NPOLE-jj+(NDATA - 1)];  

		for(jj=0; jj<slice; jj++,i++) {
			sigi[jj] = sigi[jj+slice];
		}

		if((nread=sfread(sound,(char *)(sigi+slice),
     		      slice * sfdesc.sf_class)) != slice * sfdesc.sf_class) {
			fprintf(stderr," bad read, n= %d\n",nread);
			exit(1);
		}


		if((nbytes = write(anal,(char *)coef,nblpc))!= nblpc) {
			printf(" write error, nbytes = %d\n",nbytes);
			exit(1);
		}
	}
}

alpol(sig, errn, rms1, rms2, c)
double *errn, *rms1, *rms2, *c;
int *sig;
{
	double a[POLEMAX][POLEMAX], v[POLEMAX], b[POLEMAX];
	double x[FRAMAX], y[FRAMAX];
	double *vp=v, *bp=b, *xp=x, *yp=y;
	double sum, sumx, sumy;
	int k1, i, l, k, limit, j;

	for (xp=x; xp-x < FRAME ;++xp,++sig) 
		*xp = *sig;
	k1 = NPOLE + 1;
	for (i=0; i < NPOLE ;++i)  {
		sum = (double) 0.0;
		for (k=NPOLE; k < FRAME ;++k)
			sum += x[k-(i+1)] * x[k];
		v[i] = -sum;
		if (i != NPOLE - 1)  {
			limit = NPOLE - (i+1);
			for (l=0; l < limit ;++l)  {
				sum += x[NPOLE-(i+1)-(l+1)]* x[NPOLE-(l+1)] - x[FRAME-(i+1)-(l+1)]* x[FRAME-(l+1)];
				a[(i+1)+l][l] = a[l][(i+1)+l] = sum;
			}
		}
	}
	sum = (double) 0.0;
	for (k=NPOLE; k < FRAME ;++k)
		sum += pow(x[k], (double) 2.0);
	sumy = sumx = sum;
	for (l=0; l < NPOLE ;++l)  {
		sum += pow(x[NPOLE-(l+1)], (double) 2.0) - pow(x[FRAME-(l+1)], (double) 2.0);
		a[l][l] = sum;
	}
	gauss(a, v, b);
/*	filtn(x, y, b);   */
	for (i=0; i < NPOLE ;++i)
		sumy = sumy - b[i]*v[i];
	*rms1 = sqrt(sumx/(double) (FRAME - k1 + 1) );
	*rms2 = sqrt(sumy/(double) (FRAME - k1 + 1) );
	*errn = pow(((*rms2)/(*rms1)), (double) 2.0);
	for (bp=b; bp-b < NPOLE ;++bp,++c)
		*c = *bp;
	return(0);
}

gauss(aold, bold, b)
double aold[POLEMAX][POLEMAX], *bold, *b;
{
	double amax, dum, pivot;
	double c[POLEMAX], a[POLEMAX][POLEMAX];
	int i, j, k, l, istar, ii, lp;

	/* aold and bold untouched by this subroutine */
	for (i=0; i < NPOLE ;++i)  {
		c[i] = bold[i];
		for (j=0; j < NPOLE ;++j)
			a[i][j] = aold[i][j];
	}
	/* eliminate i-th unknown */
	for (i=0; i < NPOLE - 1 ;++i)  {        /* find largest pivot */
		amax = 0.0;
		for (ii=i; ii < NPOLE ;++ii)  {
			if (fabs(a[ii][i]) >= amax)  {
				istar = ii;
				amax = fabs(a[ii][i]);
			}
		}
		if (amax < 1e-20)  {
			fprintf(stderr, "gauss: ill-conditioned\n");
			return(0);  /* ill-conditioned */
		}
		for (j=0; j < NPOLE ;++j)  {    /* switch rows */
			dum = a[istar][j];
			a[istar][j] = a[i][j];
			a[i][j] = dum;
		}
		dum = c[istar];
		c[istar] = c[i];
		c[i] = dum;
		/* pivot */
		for (j=i+1; j < NPOLE ;++j)  {
			pivot = a[j][i] / a[i][i];
			c[j] = c[j] - pivot * c[i];
			for (k=0; k < NPOLE ;++k)
				a[j][k] = a[j][k] - pivot * a[i][k];
		}
	}       /* return if last pivot is too small */
	if (fabs(a[NPOLE-1][NPOLE-1]) < 1e-20)  {
		fprintf(stderr, "gauss: ill-conditioned.\n");
		return(0);      /* ill-conditioned */
	}
	*(b+NPOLE-1) = c[NPOLE-1] / a[NPOLE-1][NPOLE-1];
	/* back substitute */
	for (k=0; k<NPOLE-1; ++k)  {
		l = NPOLE-1 -(k+1);
		*(b+l) = c[l];
		lp = l + 1;
		for (j = lp; j<NPOLE; ++j)
			*(b+l) += -a[l][j] * *(b+j);
		*(b+l) /= a[l][l];
	}
	return(1);
}


filtn(x, y, b)
double x[], b[], y[];
{
	double sum;
	int i, j;
	double *yp;

	for (yp=y; yp-y < NPOLE ;++yp)
		*yp = (double) 0.0;
	yp = y;
	for (i=NPOLE; i < FRAME; ++i)  {
		sum = x[i];
		for (j=0; j < NPOLE ;++j)  {
			sum = sum + b[j] * x[i-(j+1)];
		}
		*(yp+i) = sum;
	}
	return(0);
}
