/* numlist.c	1.6	(CARL)	7/24/84	13:00:52 */

# include <stdio.h>
# include "aswdaemon.h"
# include "numlist.h"

/*
 * routines to manage number lists
 */

/*
 * maknl makes up the head of a number
 * list and then calls addnl to add
 * the channel number to the list.
 */
struct numlist *
maknl(num, type) {
	extern char *malloc();
	register struct numlist *nl;

	if ((nl = (struct numlist *) malloc(sizeof(struct numlist))) == NULL) {
		err(ERR_NUMLIST, "maknl: out of memory");
		return(NULL);
	}

	nl->nl_num = -1;	/* number list head */
	nl->nl_type = type;
	nl->nl_forw = nl;
	nl->nl_back = nl;

	return(addnl(nl, num));
}

/*
 * append number "num" to the numberlist "nl"
 */
struct numlist *
addnl(nl, num)
	register struct numlist *nl;
{
	extern char *malloc();
	register struct numlist *tnl;

	if (nl == NULL) {
		err(ERR_NUMLIST, "addnl: nl==NULL");
		return(NULL);
	}

	if ((tnl = (struct numlist *) malloc(sizeof(struct numlist))) == NULL) {
		err(ERR_NUMLIST, "addnl: out of memory");
		return(NULL);
	}

	tnl->nl_num = num;
	tnl->nl_type = nl->nl_type;

	/* link onto end of list */
	tnl->nl_forw = nl->nl_back->nl_forw;
	tnl->nl_back = nl->nl_back;
	nl->nl_back->nl_forw = tnl;
	nl->nl_back = tnl;

	return(nl);
}

/*
 * given a numlist "onl" and a numlist "inl"
 * use inl as a list of indices and return
 * those numbers from onl.
 *
 * e.g. if onl is (5, 9, 7, 4) and inl is (0, 1)
 * then subnl would return (5, 9). if inl were
 * (2, 3) then subnl would return (7, 4). if inl
 * were (3, 0) then subnl would return (4, 5).
 *
 * indices aren't constrained to be in any particular
 * order.
 */
struct numlist *
subnl(onl, inl)
	register struct numlist *onl;	/* original numlist */
	register struct numlist *inl;	/* indices numlist */
{
	register struct numlist *nl;	/* resulting numlist */
	register struct numlist *tonl;	/* temp onl pointer */
	register struct numlist *tinl;	/* temp inl pointer */
	register int count;

	if (onl == NULL) {
		err(ERR_NUMLIST, "subnl: onl==NULL");
		return(NULL);
	}
	if (inl == NULL) {
		err(ERR_NUMLIST, "subnl: inl==NULL");
		return(NULL);
	}

	debug(DB_NUMLIST, "subnl: onl=%s, inl=%s", prnumlist(onl), prnumlist(inl));

	nl = NULL;

	/*
	 * this is grossly inefficient. there probably is
	 * a more efficient way to do this.
	 */
	for (tinl = inl->nl_forw; tinl != inl; tinl = tinl->nl_forw) {
		count = 0;
		for (tonl = onl->nl_forw; tonl != onl; tonl = tonl->nl_forw) {
			if (tinl->nl_num == count) {
				if (nl == NULL) {
					if ((nl = maknl(tonl->nl_num, onl->nl_type)) == NULL)
						return(NULL);
				}
				else {
					if ((nl = addnl(nl, tonl->nl_num)) == NULL)
						return(NULL);
				}
			}
			count++;
		}
	}

	debug(DB_NUMLIST, "subnl: nl=%s", prnumlist(nl));

	return(nl);
}

/*
 * construct a new numlist by
 * duplicating an existing one.
 */
struct numlist *
savenl(nl)
	register struct numlist *nl;
{
	register struct numlist *tnl;	/* temp nl pointer */
	register struct numlist *nnl;	/* new nl pointer */

	if (nl == NULL) {
		err(ERR_NUMLIST, "savenl: nl==NULL");
		return(NULL);
	}

	nnl = NULL;
	for (tnl = nl->nl_forw; tnl != nl; tnl = tnl->nl_forw) {
		if (nnl == NULL) {
			if ((nnl = maknl(tnl->nl_num, tnl->nl_type)) == NULL)
				return(NULL);
		}
		else {
			if ((nnl = addnl(nnl, tnl->nl_num)) == NULL)
				return(NULL);
		}
	}

	return(nnl);
}

/*
 * free a number list
 */
freenl(nl)
	register struct numlist *nl;
{
	register struct numlist *tnl;	/* temp nl pointer */
	register struct numlist *forw;	/* temp forward nl pointer */

	if (nl == NULL) {
		err(ERR_NUMLIST, "freenl: nl==NULL");
		return(RET_ERR);
	}

	/*
	 * unlink each numlist and free it
	 */
	for (tnl = nl->nl_forw; tnl != nl; tnl = forw) {
		/* unlink */
		tnl->nl_back->nl_forw = tnl->nl_forw;
		tnl->nl_forw->nl_back = tnl->nl_back;

		/* save forward link before tossing tnl */
		forw = tnl->nl_forw;

		free((char *) tnl);
	}

	/* free the head too */
	free((char *) nl);

	return(RET_OK);
}

/*
 * this allows us to set the
 * type of a numberlist. should
 * only need to set the type in
 * the head of the list, but for
 * safety's sake we'll set each
 * one.
 */
typenl(nl, type)
	register struct numlist *nl;
{
	register struct numlist *tnl;

	nl->nl_type = type;

	for (tnl = nl->nl_forw; tnl != nl; tnl = tnl->nl_forw)
		tnl->nl_type = type;
}

# define PRBUFFS	2

/*
 * return a string representation of
 * a numberlist. prnumlist returns the
 * address to a static array which is
 * overwritten on each call. ordinarily
 * something like
 *	printf("%s %s\n", prnumlist(nl1), prnumlist(nl2));
 * would fail so we use PRBUFFS separate
 * static arrays and alternate between
 * them.
 */
char *
prnumlist(nl)
	register struct numlist *nl;
{
	register struct numlist *tnl;
	static char bufs[PRBUFFS][BUFSIZ];
	static int flipflop;
	char numbuf[16];
	char *buf;
	int first;

	if (nl == NULL) {
		err(ERR_NUMLIST, "prnumlist: nl==NULL");
		return("?");
	}

	/*
	 * point buf at the next buffer. flipflop keeps track
	 * of the last one we used.
	 */
	buf = bufs[flipflop = flipflop++ % (sizeof(bufs) / sizeof(bufs[0]))];

	switch (nl->nl_type) {
		case NL_INPUT:
			strcpy(buf, "[input]");
			break;

		case NL_OUTPUT:
			strcpy(buf, "[output]");
			break;

		default:
			strcpy(buf, "[sideways]");	/* well ... */
			break;
	}

	first = 1;
	for (tnl = nl->nl_forw; tnl != nl; tnl = tnl->nl_forw) {
		sprintf(numbuf, "%d", tnl->nl_num);
		strcat(buf, first ? " " : ",");
		strcat(buf, numbuf);
		first = 0;
	}

	return(buf);
}
