/*
Contains functions which allow file selection via toolbox routines.

PathNameFrom* functions were adapted from sample code in Symantec's _Think Reference_.  Symantec got this code from an Apple Q&A Stack.
Some of the comments in these functions are speculative in that they are my attempt to explain Apple's code.

SF???File() reference:
Huxham, Fred A., David Burnard, and Jim Takatsuka.  _Using the Macintosh Toolbox With C_, 2nd ed.  San Francisco: Sybex, 1989. p. 367-73.

Assumes #inclusion of MacHeaders (this is implicit in Think C by default).
linkage: needs MacTraps library.
*/

#include <string.h>
#include <stdio.h>

/* PathNameFromDirID() is a utility function used by PathNameFromWD() */
int PathNameFromDirID(long dirID, short vRefNum, char* fullPathName)
{
	CInfoPBRec block;
	Str255 directoryName;
	OSErr err;

	*fullPathName = 0;               /* 1 */
	block.dirInfo.ioDrParID = dirID;         /* 2 */
	block.dirInfo.ioNamePtr = directoryName; /* 3 */
	do {
		block.dirInfo.ioVRefNum   = vRefNum;         /* 4 */
		block.dirInfo.ioFDirIndex = -1;              /* 4 */
		block.dirInfo.ioDrDirID   = block.dirInfo.ioDrParID; /* 4,5 */

		err = PBGetCatInfo(&block, FALSE);       /* 6 */
		if (err) return 1;

		directoryName[++directoryName[0]] = ':'; /* 7  */

		PtoCstr(directoryName);                         /* 8 */
		strcat( (char *) directoryName, fullPathName ); /* 9 */
		strcpy( fullPathName, (char *) directoryName ); /* 10 */
	} while (block.dirInfo.ioDrDirID != 2);                     /* 11 */
	return 0;
}
/*
1. Initialize to the null string
2. Parent directory = most local directory.  This is an initialization needed before loop entry.  It makes sense if you consider the most local directory the parent directory of the file in question.
3. Put directory names in "directoryName"
4. Prepare parameters to be passed in call to PBGetCatInfo();
5. Set the block's DrDirID to that of its parent.  This is how we travel up the directory tree.
6. Get catalog information for this block.  In our case, the only info we use is the string at ioNamePtr (directoryName) and ioDrParID.
7. Append ':' onto the new directory name & increment string length byte directoryName will be of the form "DirName:"
8. Convert Pascal --> C string format
9. Append the lower part of the path onto the new directory name. fullPathName will be of the form "DirName2:DirName1:DirName0:"
10. Make a copy of directoryName in fullPathName since directoryName will be overwritten on next pass through PBGetCatInfo
11. Stop when the highest level directory (the volume) has been reached
*/

/*
PathNameFromWD() puts the full pathname of the HFS working directory "vRefNum" into "pathName".
It does this by calling PBGetWDInfo to get the VRefNum and DirID of the real directory. It then calls PathNameFromDirID, and returns its result.
*/
int PathNameFromWD(long vRefNum, char* pathName)
{
	WDPBRec myBlock;
	OSErr err;

	myBlock.ioNamePtr  = nil;     /* 1 */
	myBlock.ioVRefNum  = vRefNum; /* 1 */
	myBlock.ioWDIndex  = 0;       /* 1 */
	myBlock.ioWDProcID = 0;       /* 1 */
	
	err = PBGetWDInfo(&myBlock, FALSE); /* 2 */
	if (err) return 1;

	return
		PathNameFromDirID(myBlock.ioWDDirID, myBlock.ioWDVRefNum, pathName);
}
/*
1. Prepare parameters to be passed in call to PBGetWDInfo()
2. Change the Working Directory number in vRefnum into a real
   vRefnum and DirID. The real vRefnum is returned in ioVRefnum,
   and the real DirID is returned in ioWDDirID.
*/

/*
getfullfstr() gets a full input file name from a dialog box selection. (1)
It displays files of all "numtyp" types specified in "types".
If numtyp == -1, files of all types are displayed.
*/
int getfullfstr( Point where, int numtyp, SFTypeList types, char *fstr )
{
	SFReply reply;
	int err;

	printf("\n"); /* 2 */
	SFGetFile( where, "\p" , NULL, numtyp, types, NULL, &reply);

	if (!reply.good) return 1; /* 3 */

	err = PathNameFromWD( reply.vRefNum, fstr ); /* 4 */
	if (err) return 1;
	strcat( fstr, PtoCstr( reply.fName ) ); /* 5 */
	printf( "Input File Chosen:\n%s\n", fstr );
	return 0;
}

/*
putfullfstr() gets a full output file name from a dialog box selection. (1)
It displays prompt "prompt" and default file name "fstr" in the dialog box.
*/
int putfullfstr( Point where, char *prompt, char *fstr )
{
	SFReply reply;
	int err;

	printf("\n"); /* 2 */
	SFPutFile( where, CtoPstr(prompt) , CtoPstr(fstr), NULL, &reply);

	if (!reply.good) return 1; /* 3 */

	err = PathNameFromWD( reply.vRefNum, fstr ); /* 4 */
	if (err) return 1;
	strcat( fstr, PtoCstr( reply.fName ) ); /* 5 */
	printf( "Output File Chosen:\n%s\n", fstr );
	return 0;
}
/*
1. The appropriate dialog box is placed at screen location "where".
   The full name of the file chosen is returned in "fstr".
   The function returns 1 on error and zero on no error.
   It is strongly recommended that "fstr" be 256 bytes long, though one could usually get away with a length of about 80.
2. For reasons unclear to me, it is necessary to write to the console for SF???File() to work.  An error sends me into MacsBug otherwise.
3. User clicking "Cancel" is considered an error.
4. Put full path name of directory into fstr.
5. Append file name to full path name to yield full file name.
*/

