*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE PFYLIM (Y, NPT, YMIN, YMAX, NCOF, SFREQ,
*                        IPLOT, FLTTYP)
*
*
* Purpose:
*     Find the ordinate limits for a filter response plot
*
*
* Description:
*     This routine finds plot limits for a filter response plot.
*
*
* Parameters:
* R ->  Y      - Ordinate values
* I ->  NPT    - Number of values
* R <-  YMIN   - Lower plot limit
* R <-  YMAX   - Upper plot limit
* I ->  NCOF   - Number of filter coefficients
* R ->  SFREQ  - Sampling frequency
* I ->  IPLOT  - Type of plot,
*                1 - linear amplitude
*                2 - dB amplitude
*                3 - log amplitude
*                4 - group delay
*                5 - phase response
*                6 - radian phase
*                7 - impulse response
* C ->  FLTTYP - Character string indicating the type of filter,
*                ALL - all-pole filter
*                CAS - cascade analog filter
*                FIR - finite impulse response filter
*                IIR - infinite impulse response filter
*                WIN - window response
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.4 $  $Date: 1995/03/08 15:46:30 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE PFYLIM (Y, NPT, YMIN, YMAX, NCOF, SFREQ,
     -                   IPLOT, FLTTYP)


      INTEGER TICKV,EXACTV,LINS,LOGS
      PARAMETER (TICKV=1,EXACTV=2,LINS=0,LOGS=1)
      INTEGER LINAR,DBAR,LOGAR,DELYR,PHASD,PHASR,IMPLR
      PARAMETER (LINAR=1,DBAR=2,LOGAR=3,DELYR=4,PHASD=5,PHASR=6,
     -           IMPLR=7)
      REAL PI
      PARAMETER (PI=3.141 592 654)

      INTEGER NPT,NCOF,IPLOT
      INTEGER ILL,IUL,ILOG,NSECT

      CHARACTER*(*) FLTTYP

      REAL Y(0:NPT-1)
      REAL YMIN,YMAX,SFREQ
      REAL DMIN,DMAX
      REAL CEILX,FLOORX


* Find the minimum and maximum ordinate values
      CALL MINMAX(Y,NPT,DMIN,DMAX)

* Linear plot scale, include 0
      IF (IPLOT.EQ.LINAR) THEN
        DMIN=MIN(DMIN,0.0)
        ILL=TICKV
        IUL=TICKV
        ILOG=LINS

* DB plot scale (at least 10 dB but at most 100 dB range)
      ELSE IF (IPLOT.EQ.DBAR) THEN
        DMAX=10.*CEILX(0.1*(DMAX+1.))
        IUL=EXACTV
        DMIN=10.*FLOORX(0.1*(DMIN-1.))
        DMIN=MAX(DMAX-100.0,MIN(DMIN,DMAX-10.0))
        ILL=EXACTV
        ILOG=LINS

* Log plot scale (minimum 1 decade, maximum 5 decades)
      ELSE IF (IPLOT.EQ.LOGAR) THEN
        IUL=TICKV
        DMIN=MAX(DMIN,1E-4*DMAX)
        ILL=TICKV
        ILOG=LOGS

* Group delay, limit the range
      ELSE IF (IPLOT.EQ.DELYR) THEN
        IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN') THEN
          DMIN=0.0
          ILL=EXACTV
          DMAX=MIN(-5*(NCOF-1)/SFREQ,DMAX,0.0)
          IUL=TICKV
        ELSE IF (FLTTYP.EQ.'ALL') THEN
          DMIN=MAX(5*(NCOF-1)/SFREQ,DMIN,0.0)
          ILL=TICKV
          DMAX=0.0
          IUL=EXACTV
        ELSE
          NSECT=NCOF/5
          DMIN=MAX(DMIN,-10*NSECT/SFREQ)
          ILL=TICKV
          DMAX=MIN(DMAX,10*NSECT/SFREQ)
          IUL=TICKV
        END IF
        ILOG=LINS

* Phase response, limit the range to +/- 360 degrees per section
      ELSE IF (IPLOT.EQ.PHASD) THEN
        IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN' .OR.
     -      FLTTYP.EQ.'ALL') THEN
          DMIN=-180.
          ILL=EXACTV
          DMAX=+180
          IUL=EXACTV
        ELSE
          NSECT=NCOF/5
          DMAX=180.*CEILX(MIN(DMAX,NSECT*360.)/180.)
          IUL=EXACTV
          DMIN=180.*FLOORX(MAX(DMIN,-NSECT*360.)/180.)
          DMIN=MIN(DMIN,DMAX-180.)
          ILL=EXACTV
        END IF
        ILOG=LINS

* Phase response, limit the range to +/- 2 Pi per section
      ELSE IF (IPLOT.EQ.PHASR) THEN
        IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN' .OR.
     -      FLTTYP.EQ.'ALL') THEN
          DMIN=-1.
          ILL=EXACTV
          DMAX=+1.
          IUL=EXACTV
        ELSE
          NSECT=NCOF/5
          DMAX=CEILX(MIN(DMAX,2.*NSECT))
          IUL=EXACTV
          DMIN=FLOORX(MAX(DMIN,-2.*NSECT))
          DMIN=MIN(DMIN,DMAX-1.)
          ILL=EXACTV
        END IF
        ILOG=LINS

* Impulse response, include zero
      ELSE IF (IPLOT.EQ.IMPLR) THEN
        DMIN=MIN(DMIN,0.0)
        ILL=TICKV
        DMAX=MAX(DMAX,0.0)
        IUL=TICKV
        ILOG=LINS

* Unknown plot option
      ELSE
        CALL HALT('PFYLIM - Invalid plot option')

      END IF

* Find the plot limits
      CALL PLTLIM(DMIN,DMAX,ILL,IUL,ILOG,YMIN,YMAX)


      RETURN

      END
