*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE PFSETX (IPLOT, IFAXIS, FLTTYP, NCOF, SFREQ,
*                        X, MXNPT, NPT,
*                        XMIN, XMAX, XSPEC)
*
*
* Purpose:
*     Set the array of abscissa values for a filter response
*
*
* Description:
*     This routine sets the X-array for a filter response plot.
*
*
* Parameters:
* I ->  IPLOT  - Type of plot,
*                1 - linear amplitude
*                2 - dB amplitude
*                3 - log amplitude
*                4 - group delay
*                5 - phase response
*                6 - radian phase
*                7 - impulse response
* I ->  IFAXIS - Frequency axis options,
*                0 - linear frequency (Hz)
*                1 - log frequency (Hz)
*                2 - linear radian frequency
*                3 - log radian frequency
* C ->  FLTTYP - Character string indicating the type of filter,
*                ALL - all-pole filter
*                CAS - cascade analog filter
*                FIR - finite impulse response filter
*                IIR - infinite impulse response filter
*                WIN - window response
* I ->  NCOF   - Number of filter coefficients
* R ->  SFREQ  - Sampling frequency
* R <-  X      - Array of abscissa values
* I ->  MXNPT  - Maximum number of abscissa values
* I <-  NPT    - Number of abscissa values
* R <-> XMIN   - Lower plot limit.  This is both an input and an output
*                variable if XSPEC is .TRUE., and an output variable if
*                XSPEC is .FALSE..
* R <-> XMAX   - Upper plot limit.  This is both an input and an output
*                variable is XSPEC is .TRUE., and an output variable if
*                XSPEC is .FALSE..
* L ->  XSPEC  - Logical variable,
*                .TRUE.  - XMIN and XMAX specified on input
*                .FALSE. - XMIN and XMAX not specified on input
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.7 $  $Date: 1995/03/08 15:46:21 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE PFSETX (IPLOT, IFAXIS, FLTTYP, NCOF, SFREQ,
     -                   X, MXNPT, NPT,
     -                   XMIN, XMAX, XSPEC)


      INTEGER TICKV,EXACTV
      PARAMETER (TICKV=1,EXACTV=2)
      INTEGER LINAR,DBAR,LOGAR,DELYR,PHASD,PHASR,IMPLR
      PARAMETER (LINAR=1,DBAR=2,LOGAR=3,DELYR=4,PHASD=5,PHASR=6,
     -           IMPLR=7)
      INTEGER LINS,LOGS,CFREQ,RFREQ
      PARAMETER (LINS=0,LOGS=1,CFREQ=0,RFREQ=1)
      INTEGER MNNPT
      PARAMETER (MNNPT=401)
      REAL PPC
      PARAMETER (PPC=32.0)

      INTEGER IPLOT,IFAXIS,NCOF,MXNPT,NPT
      INTEGER ILOG,IRAD,ILL,IUL,I,NSECT
      INTEGER ISS,IFS
      INTEGER IFLORX,ICEILX

      LOGICAL XSPEC

      CHARACTER*(*) FLTTYP

      REAL SFREQ,XMIN,XMAX
      REAL DMIN,DMAX,R,RESL
      REAL X(0:MXNPT-1)


* Set up the default limits
*  1. Time axis for impulse responses (FIR)
*     lower limit is 0
*     upper limit is (NCOF-1)/SFREQ
*  2. Time axis for impulse responses (IIR)
*     lower limit is 0
*     upper limit is set to (MXNPT-1)/SFREQ
*  3. Linear frequency axis (digital filters)
*     lower limit is 0
*     upper limit is 0.5*Sfreq
*  4. Log frequency axis (digital filters)
*     lower limit is tick value below 0.5*Sfreq/1000
*     upper limit is 0.5*Sfreq
*  5. Linear frequency axis (analog filters)
*     lower limit is 0
*     upper limit is 5 Hz
*  6. Log frequency axis (analog filters)
*     lower limit is tick value below 5/1000 Hz
*     upper limit is tick value above 5 Hz

*==========
* Time axis
      IF (IPLOT.EQ.IMPLR) THEN

        IF (XSPEC) THEN
          DMIN=XMIN
          DMAX=XMAX
          XMIN=MIN(DMIN,DMAX)
          XMAX=MAX(DMIN,DMAX)
          ISS=IFLORX(XMIN*SFREQ)
          IFS=ICEILX(XMAX*SFREQ)
          NPT=IFS-ISS+1
          IF (NPT.GT.MXNPT) THEN
            CALL WARN('PFSETX - Too many response samples requested')
            NPT=MXNPT
          END IF

        ELSE
* Choose the default number of points as either the number of filter
* coefficients (FIR filters) or the maximum number of points (IIR
* filters)
          IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN') THEN
            NPT=NCOF
            IF (NPT.GT.MXNPT) THEN
              CALL WARN('PFSETX - Impulse response truncated')
              NPT=MXNPT
            END IF
          ELSE
            NPT=MXNPT
          END IF
          XMIN=0.0
          ISS=0
          XMAX=(NPT-1)/SFREQ

        END IF

* Fill in the X-array
        DO 100 I=0,NPT-1
          X(I)=(I+ISS)/SFREQ
 100    CONTINUE

*==========
* Frequency axis
      ELSE
        ILOG=MOD(IFAXIS,2)
        IRAD=MOD(IFAXIS/2,2)

        IF (XSPEC) THEN
          ILL=EXACTV
          IUL=EXACTV

        ELSE
          IF (FLTTYP.EQ.'CAS') THEN
            IF (ILOG.EQ.LOGS) THEN
              XMAX=5.0
              IUL=TICKV
              XMIN=XMAX/1000.
              ILL=TICKV
            ELSE
              XMIN=0.0
              ILL=EXACTV
              XMAX=5.0
              IUL=TICKV
            END IF
          ELSE
            IF (ILOG.EQ.LOGS) THEN
              XMAX=0.5*SFREQ
              IUL=EXACTV
              XMIN=XMAX/1000.
              ILL=TICKV
            ELSE
              XMIN=0.0
              ILL=EXACTV
              XMAX=0.5*SFREQ
              IUL=EXACTV
            END IF
          END IF

        END IF

* Find suitable abscissa limits
        IF (IRAD.EQ.CFREQ) THEN
          CALL PLTLIM(XMIN,XMAX,ILL,IUL,ILOG,DMIN,DMAX)
          XMIN=DMIN
          XMAX=DMAX
        ELSE
          CALL PLTLIM(2.*XMIN,2.*XMAX,ILL,IUL,ILOG,DMIN,DMAX)
          XMIN=0.5*DMIN
          XMAX=0.5*DMAX
        END IF
        IF (ILOG.EQ.LOGS) THEN
          R=XMAX/XMIN
        ELSE
          R=1.
        END IF

* Determine the number of abscissa values
* The number of values is at least MNNPT and at most MXNPT.  The number
* of points is calculated below for linear frequency scales, but is
* also used for log frequency scales.
* FIR filters:
* - The case of two maximally separated impulses gives rise to a
*   sinusoidal variation in the frequency response of the form
*   cos (2 pi f (N-1)/2).  The frequency resolution should be at
*   least PPC points per cycle.
* All-pole filters:
* - The response is the inverse to that for an FIR filter.  The
*   frequency resolution is set to be the same as for an FIR filter
*   with the same number of coefficients.
* IIR filters:
* - The denominator of the overall system function has NSECT*3-1
*   coefficients.  The resolution is set to be the same as an all-pole
*   filter with that many cofficients.
* CAS filters:
* - The denominator of the overall system function has NSECT*3-1
*   coefficients.  For such filters, there is no convenient estimate of
*   the resolution needed.  It is assumed that the filter response is
*   similar to that for a digital IIR filter spanning the range from 0
*   to half the sampling frequency.
        IF (FLTTYP.EQ.'IIR') THEN
          NSECT=NCOF/5
          RESL=2.0*SFREQ/((NSECT*3-1)*PPC)
          NPT=ICEILX((XMAX-XMIN)/RESL)+1
        ELSE IF (FLTTYP.EQ.'CAS') THEN
          NSECT=NCOF/5
          RESL=2.0/((NSECT*3-1)*PPC)
          NPT=ICEILX(0.5/RESL)+1
        ELSE
          RESL=2.0*SFREQ/((NCOF-1)*PPC)
          NPT=ICEILX((XMAX-XMIN)/RESL)+1
        END IF
        NPT=MAX(MNNPT,NPT)
        NPT=MIN(MXNPT,NPT)

* Fill in the X-array
        CALL GEOINC(XMIN,XMAX,NPT,R,X)

      END IF


      RETURN

      END
