*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE ORSCAL (Y, NPT, YMIN, YMAX, SFREQ, IPLOT,
*                         YLABEL, NCYL, YUNITS, NCYU)
*
*
* Purpose:
*     Determine an ordinate scale for a filter response plot
*
*
* Description:
*     If the data represents group delay, the scale is expressed in
*     scaled seconds if possible.  In addition, the data array is
*     clipped to [ YMIN-1000(YMAX-YMIN), YMAX+1000(YMAX-YMIN) ] if
*     YMAX > YMIN.
*
*
* Parameters:
* R <-> Y      - Ordinate data array, scaled and clipped if necessary
* I ->  NPT    - Number of data values
* R <-> YMIN   - Minimum ordinate, scaled if necessary
* R <-> YMAX   - Maximum ordinate, scaled if necessary
* R ->  SFREQ  - Sampling frequency
* I ->  IPLOT  - Type of plot,
*                1 - linear amplitude
*                2 - dB amplitude
*                3 - log amplitude
*                4 - group delay
*                5 - phase response
*                6 - radian phase
*                7 - impulse response
* C <-  YLABEL - "Amplitude" (IPLOT=1,2,3, or 7)
*                "Group Delay" (IPLOT=4)
*                "Phase Response" (IPLOT=5 or 6)
* I <-  NCYL   - Number of characters in YLABEL (max 18)
* C <-  YUNITS - "", "dB", "samples", "sec", "ms", "us",
*                "ns", "ps", "P rad", or "degrees"
* I <-  NCYU   - Number of characters in YUNITS (max 7)
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.5 $  $Date: 1995/03/08 15:46:12 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE ORSCAL (Y, NPT, YMIN, YMAX, SFREQ, IPLOT,
     -                   YLABEL, NCYL, YUNITS, NCYU)


      INTEGER TICKV,EXACTV
      PARAMETER (TICKV=1,EXACTV=2)
      INTEGER LINAR,DBAR,LOGAR,DELYR,PHASD,PHASR,IMPLR
      PARAMETER (LINAR=1,DBAR=2,LOGAR=3,DELYR=4,PHASD=5,PHASR=6,
     -          IMPLR=7)
      INTEGER LINS,LOGS
      PARAMETER (LINS=0,LOGS=1)

      INTEGER NPT,IPLOT,NCYL,NCYU
      INTEGER ILL,IUL,ILOG,IEXP
      INTEGER IEXPSC,LENSTR

      CHARACTER*(*) YLABEL,YUNITS
      CHARACTER*1 CPI,CMU

      REAL Y(NPT)
      REAL YMIN,YMAX,SFREQ
      REAL VMIN,VMAX,SF,YR


* Get the exponent of the scaling factor
      VMIN=MIN(YMIN,YMAX)
      VMAX=MAX(YMIN,YMAX)
      ILL=EXACTV
      IUL=EXACTV
      IF (VMIN.GE.VMAX) THEN
        CALL AMINMX(Y,NPT,VMIN,VMAX)
        ILL=TICKV
        IUL=TICKV
      END IF
      IF (IPLOT.EQ.LOGAR) THEN
        ILOG=LOGS
      ELSE
        ILOG=LINS
      END IF
      IEXP=IEXPSC(VMIN,VMAX,ILL,IUL,ILOG)

* Linear amplitude
      IF (IPLOT.EQ.LINAR) THEN
        YLABEL='Amplitude'
        YUNITS=' '
        SF=1.

* DB amplitude
      ELSE IF (IPLOT.EQ.DBAR) THEN
        YLABEL='Amplitude'
        YUNITS='dB'
        SF=1.

* Log amplitude
      ELSE IF (IPLOT.EQ.LOGAR) THEN
        YLABEL='Amplitude'
        YUNITS=' '
        SF=1.

* Group delay
      ELSE IF (IPLOT.EQ.DELYR) THEN
        YLABEL='Group Delay'
        IF (SFREQ.EQ.1.0) THEN
          YUNITS='samples'
          SF=1.
        ELSE IF (IEXP.EQ.-3) THEN
          YUNITS='ms'
          SF=1E+3
        ELSE IF (IEXP.EQ.-6) THEN
          CMU=CHAR(181)
          YUNITS=CMU//'s'
          SF=1E+6
        ELSE IF (IEXP.EQ.-9) THEN
          YUNITS='ns'
          SF=1E+9
        ELSE IF (IEXP.EQ.-12) THEN
          YUNITS='ps'
          SF=1E+12
        ELSE
          YUNITS='sec'
          SF=1.
        END IF

* Phase response (degrees)
      ELSE IF (IPLOT.EQ.PHASD) THEN
        YLABEL='Phase Response'
        YUNITS='degrees'
        SF=1.

* Phase response (radians)
      ELSE IF (IPLOT.EQ.PHASR) THEN
        YLABEL='Phase Response'
        CPI=CHAR(182)
        YUNITS=CPI//' rad'
        SF=1.

* Impulse response
      ELSE IF (IPLOT.EQ.IMPLR) THEN
        YLABEL='Amplitude'
        YUNITS=' '
        SF=1.

* Unknown plot option
      ELSE
        CALL HALT('ORSCAL - Invalid plot option')

      END IF

* Scale the data
      IF (SF.NE.1.) THEN
        YMIN=SF*YMIN
        YMAX=SF*YMAX
        CALL VSMULN(Y,SF,Y,NPT)
      END IF

* Clip the data
      IF (YMAX.GT.YMIN) THEN
        YR=1000.*(YMAX-YMIN)
        CALL VCLIP(Y,Y,NPT,YMIN-YR,YMAX+YR)
      END IF

* See if a power of ten annotation will be added
      IEXP=IEXPSC(SF*VMIN,SF*VMAX,ILL,IUL,ILOG)

* Return the number of characters in the strings
      NCYL=LENSTR(YLABEL)
      IF (IEXP.NE.0) NCYL=NCYL+6
      NCYU=LENSTR(YUNITS)


      RETURN

      END
