*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE PLAXES (IXAX, ATICKX, ALABLX, NTICKX, IEXPX,
*                        IYAX, ATICKY, ALABLY, NTICKY, IEXPY,
*                        DATREG)
*
*
* Purpose:
*     Draw axes around a graph
*
*
* Description:
*     This routine draws axes around a rectangular plotting area.
*     Additional axes through the data origin can be drawn if the
*     range f data includes the origin.  The axes may optionally be
*     labelled.  This routine uses the character size in effect to
*     determine the character size for the labels.  In addition, axis
*     tick marks scale with the character size.
*
*
* Parameters:
* I ->  IXAX   - Three element integer array specifying the X-axis
*                options
*                IXAX(1) - Bottom axis options
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear below the
*                              axis
*                IXAX(2) - Origin axis options. The origin axis is only
*                          drawn if Y=0 is included in the range of
*                          data.
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear below the
*                              axis
*                IXAX(3) - Top axis options
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear below the
*                              axis
* R ->  ATICKX - Array of data values at which ticks are to be placed
*                for a horizontal axis.  If a data value lies outside
*                the range spanned by the axis [DATREG(1,1) to
*                DATREG(1,2)], the corresponding tick is not drawn.
* C ->  ALABLX - Character array of labels to be centred above or below
*                the horizontal axis tick marks.  For the purpose of
*                determining the length of the label, a null character
*                if present, marks the end of the string.  The label
*                array is in one-to-one correspondence with the array
*                of tick values.  Even for an unlabelled axis, whether
*                a label has zero length or not affects the size of the
*                corresponding axis tick.
* I ->  NTICKX - Number of horizontal axis tick values
* I ->  IEXPX  - Exponent for the horizontal axes.  If IEXPX is
*                non-zero, a power of ten annotation is printed for
*                each labelled horizontal axis.
* I ->  IYAX   - Three element integer array specifying the Y-axis
*                options
*                IYAX(1) - Left axis options
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear to the left
*                              of the axis
*                IYAX(2) - Origin axis options. The origin axis is only
*                          drawn if X=0 is included in the range of
*                          data.
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear to the left
*                              of the axis
*                IYAX(3) - Right axis options
*                          0 - not drawn
*                          1 - unlabelled axis without tick marks
*                          2 - unlabelled axis with tick marks
*                          3 - labelled axis, labels appear to the
*                              right of the axis
* R ->  ATICKY - Array of data values at which ticks are to be placed
*                for a vertical axis.  If a data value lies outside
*                the range spanned by the axis [DATREG(2,1) to
*                DATREG(2,2)], the corresponding tick is not drawn.
* C ->  ALABLY - Character array of labels to be centred beside the
*                vertical axis tick marks.  For the purpose of
*                determining the length of the label, a null character
*                if present, marks the end of the string.  The label
*                array is in one-to-one correspondence with the array
*                of tick values.  Even for an unlabelled axis, whether
*                a label has zero length or not affects the size of the
*                corresponding axis tick.
* I ->  NTICKY - Number of vertical axis tick values
* I ->  IEXPY  - Exponent for the vertical axes.  If IEXPY is non-zero,
*                a power of ten annotation is printed for each labelled
*                vertical axis.
* R ->  DATREG - Data limits.  The four elements of DATREG define the
*                plot region.
*                DATREG(1,1) - Lower limit for the X data
*                DATREG(2,1) - Lower limit for the function values
*                DATREG(1,2) - Upper limit for the X data
*                DATREG(2,2) - Upper limit for the function values
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.4 $  $Date: 1995/03/08 15:28:09 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE PLAXES (IXAX, ATICKX, ALABLX, NTICKX, IEXPX,
     -                   IYAX, ATICKY, ALABLY, NTICKY, IEXPY,
     -                   DATREG)


      INTEGER OFF,LINE,ON,LABEL,ILEFT,IRIGHT,IBOTH
      PARAMETER (OFF=0,LINE=1,ON=2,LABEL=3,ILEFT=-1,IRIGHT=+1,IBOTH=0)

      INTEGER IXAX(3),IYAX(3)
      INTEGER NTICKX,NTICKY,IEXPX,IEXPY
      INTEGER IXORIG,IYORIG,LENX,LENY
      INTEGER IAX(3),IOFFS(3),IFN(3)
      INTEGER I,IY,IX,IW,IH

      CHARACTER*(*) ALABLX(*),ALABLY(*)

      REAL ATICKX(*),ATICKY(*),DATREG(2,2)
      REAL YD(3),XD(3)
      REAL ANG,PVX,PVY

      DATA IFN/IRIGHT,IBOTH,ILEFT/
      DATA YD/-6.666666667,-6.666666667,4.6666666667/
      DATA XD/-6.666666667,-6.666666667,8.6666666667/

* Note on the power of ten label
*   For the x-axis, the lower right hand corner of the power
*   of ten label is 6 double width character positions from the
*   right hand edge of the axis (5*1.5*(2*CWID)+2*CWID=17*CWID)

* Get the character size
      CALL GETCHR(IW,IH,ANG)

* Find the plot scaling
      CALL GETGRS(IXORIG,IYORIG,LENX,LENY)
      PVX=LENX/(DATREG(1,2)-DATREG(1,1))
      PVY=LENY/(DATREG(2,2)-DATREG(2,1))

* Determine which horizontal axes are to appear on the plot
      IAX(1)=IXAX(1)
      IAX(2)=IXAX(2)
      IAX(3)=IXAX(3)
      IOFFS(1)=0
      IOFFS(2)=NINT(-DATREG(2,1)*PVY)
      IOFFS(3)=LENY
      IF (IOFFS(2).LT.0 .OR. IOFFS(2).GT.LENY) THEN
        IAX(2)=OFF
      ELSE IF (IOFFS(2).EQ.0) THEN
        IAX(1)=MAX(IAX(1),IAX(2))
        IAX(2)=OFF
      ELSE IF (IOFFS(2).EQ.LENY) THEN
        IAX(3)=MAX(IAX(2),IAX(3))
        IAX(2)=OFF
      END IF

* Draw the horizontal axes
      CALL SETCHR(IW,IH,0.0)
      DO 120 I=1,3
        IY=IYORIG+IOFFS(I)
        IF (IAX(I).EQ.LINE) THEN
          CALL XYVECT(IXORIG,IY,IXORIG+LENX,IY)
        ELSE IF (IAX(I).EQ.ON) THEN
          CALL DRWHAX(IXORIG,IY,LENX,DATREG(1,1),DATREG(1,2),
     -                ATICKX,ALABLX,-NTICKX,IFN(I))
        ELSE IF (IAX(I).EQ.LABEL) THEN
          CALL DRWHAX(IXORIG,IY,LENX,DATREG(1,1),DATREG(1,2),
     -                ATICKX,ALABLX,NTICKX,IFN(I))
          IF (IEXPX.NE.0)
     -      CALL SFPW10(IXORIG+LENX-17*IW,
     -                  IY+NINT(YD(I)*IH),6,IEXPX)
        END IF
 120  CONTINUE


* Determine which vertical axes are to appear on the plot
      IAX(1)=IYAX(1)
      IAX(2)=IYAX(2)
      IAX(3)=IYAX(3)
      IOFFS(1)=0
      IOFFS(2)=NINT(-DATREG(1,1)*PVX)
      IOFFS(3)=LENX
      IF (IOFFS(2).LT.0 .OR. IOFFS(2).GT.LENX) THEN
        IAX(2)=OFF
      ELSE IF (IOFFS(2).EQ.0) THEN
        IAX(1)=MAX(IAX(1),IAX(2))
        IAX(2)=OFF
      ELSE IF (IOFFS(2).EQ.LENX) THEN
        IAX(3)=MAX(IAX(2),IAX(3))
        IAX(2)=OFF
      END IF

* Draw the vertical axes
      CALL SETCHR(IW,IH,90.)
      DO 220 I=1,3
        IX=IXORIG+IOFFS(I)
        IF (IAX(I).EQ.LINE) THEN
          CALL XYVECT(IX,IYORIG,IX,IYORIG+LENY)
        ELSE IF (IAX(I).EQ.ON) THEN
          CALL DRWVAX(IX,IYORIG,LENY,DATREG(2,1),DATREG(2,2),
     -                ATICKY,ALABLY,-NTICKY,IFN(I))
        ELSE IF (IAX(I).EQ.LABEL) THEN
          CALL DRWVAX(IX,IYORIG,LENY,DATREG(2,1),DATREG(2,2),
     -                ATICKY,ALABLY,NTICKY,IFN(I))
          IF (IEXPY.NE.0)
     -      CALL SFPW10(IX+NINT(XD(I)*IH),
     -                  IYORIG+LENY-17*IW,6,IEXPY)
        END IF
 220  CONTINUE

* Restore the character plotting angle
      CALL SETCHR(IW,IH,ANG)


      RETURN

      END
