*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE AXTICK (FIRSTV, DELTAV, NTICK, ATICK, ALABEL, IEXP)
*
*
* Purpose:
*     Generate axis ticks and labels for a linear plot
*
*
* Description:
*     This routine fills in an array of axis values corresponding to
*     locations for axis ticks.  The tick values are uniformly spaced
*     given an initial value and an increment.  The character labels
*     corresponding to these ticks are also generated.  Normally only
*     every second tick has a non-null label.  The label strings are
*     terminated with a null character if they are shorter than the
*     length of the output character string.
*
*
* Parameters:
* R ->  FIRSTV - Data value corresponding to the first tick
* R ->  DELTAV - Data value increments between ticks.  DELTAV must be
*                greater than zero.
* I ->  NTICK  - Number of tick values
* R <-  ATICK  - Array of data values corresponding to the ticks
* C <-  ALABEL - Character array of labels corresponding to the ticks.
*                These are in one-to-one correspondence with the array
*                of tick values.  Each element of ALABEL should be at
*                least 6 characters in length.  Each of the generated
*                labels is terminated with a null character if it is
*                shorter than the length of the array elements.
* I <-  IEXP   - Exponent for the axis labels.  The value of the tick
*                corresponding to an axis label is the value coded in
*                the label string times 10**IEXP.  IEXP is returned as
*                a multiple of 3.  Note that IEXP will be zero if all
*                of the labels can be represented in 4 significant
*                digits.
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.5 $  $Date: 1995/03/08 15:27:37 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE AXTICK (FIRSTV, DELTAV, NTICK, ATICK, ALABEL, IEXP)


      INTEGER NPL
      REAL EPS
      PARAMETER (EPS=1E-4,NPL=6)

      INTEGER NTICK,IEXP
      INTEGER ISTRT,INC,IE,ID,IT,I,NC
      INTEGER IFLORX

      CHARACTER*(NPL) CSTR
      CHARACTER*(*) ALABEL(*)

      REAL FIRSTV,DELTAV
      REAL ATICK(*)
      REAL ALASTV,PS


* Label every tick if the number of intervals is small
      IF (NTICK.LE.5) THEN
        ISTRT=1
        INC=1

      ELSE

* Only every second tick will be labelled.  Determine which ticks
* should be labelled.  In the usual case, FIRSTV is a multiple of
* DELTAV, where DELTAV is of the form 1, 2 or 5 times a power of 10.
* There are two cases: either ticks are labelled from the first
* one or ticks are labelled from the second one.
* 1) Normally, the ticks corresponding to even multiples of DELTAV
*    will be labelled.  This ensures that the origin if is within
*    range of the axis will be labelled.
* 2) If the number of ticks is odd, labelling starting with the
*    first tick will give one more labelled tick than starting
*    with the second tick.  If the origin does not appear within the
*    range of the axis AND DELTAV is an even number times a power
*    of 10 (viz. 2) AND the number of ticks is odd, labelling
*    starts with the first tick.
        ALASTV=FIRSTV+(NTICK-1)*DELTAV
        IE=IFLORX(LOG10((1.+EPS)*DELTAV))
        ID=(1.+EPS)*DELTAV/10.**IE
        IT=(1.+EPS)*ABS(FIRSTV/DELTAV)
        IF (MOD(IT,2).EQ.0 .OR.
     -      (MOD(ID,2).EQ.0 .AND. FIRSTV*ALASTV.GT.0.0 .AND.
     -       MOD(NTICK,2).NE.0)) THEN
          ISTRT=1
        ELSE
          ISTRT=2
        END IF
        INC=2

      END IF

* Find a scale factor for the axis labels
      CALL ENGSCL(FIRSTV+(ISTRT-1)*DELTAV,INC*DELTAV,
     -            (NTICK+INC-ISTRT)/INC,NPL-2,IEXP)
      PS=10.**(-IEXP)

* Generate the tick positions and the labels
      DO 400 I=1,NTICK

        ATICK(I)=FIRSTV+(I-1)*DELTAV

* Convert the tick value to a character string
        IF (MOD(I-ISTRT,INC).EQ.0) THEN
          CALL LABEL(PS*ATICK(I),NPL,CSTR,NC)
          ALABEL(I)=CSTR(1:NC)//CHAR(0)
        ELSE
          ALABEL(I)=CHAR(0)
        END IF

 400  CONTINUE


      RETURN

      END
