*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE EQDINC (X, N, XI, MAXNI, NI)
*
*
* Purpose:
*     Generate equidistant points between given values
*
*
* Description:
*     This subroutine generates an output array with an ordered set of
*     values which are equidistantly spaced in the intervals between
*     values of the input array.  Samples are also generated to
*     tcoincide with the input values.  For the interval k between X(k)
*     and X(k+1), the number of output values is given by (for a fixed
*     value DX)
*
*       Mk = ceil(DXk/DX) , where DXk=X(k+1)-X(k) .
*
*     Mk includes the first point of the interval (which coincides with
*     X(k)), but not the last point of the interval (which coincides
*     with X(k+1)).  The total number of sample points is given by
*
*                N-1        X(N)-X(1)
*       Ni = 1 + SUM Mk  <= --------- + N = Nimax .
*                k=1           DX
*
*     The upper bound is used to solve for an appropriate value for DX.
*     This value DX is the minimum spacing between sample values.  This
*     value of DX is then used to generate the actual sample points.
*     The number of output points then lies in the range given by
*
*       Nimax - (N-1) <= Ni <= Nimax .
*
*
* Parameters:
* R ->  X      - Input array of data values.  These values must be in
*                increasing order
* I ->  N      - Number of input data values
* R <-  XI     - Output array of sample values.  These values will be
*                in increasing order.
* I ->  MAXNI  - Maximum number of sample values.  MAXNI must be at
*                least N.
* I <-  NI     - Actual number of sample values generated,
*                MAXNI-(N-1) <= NI <= MAXNI.
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.5 $  $Date: 1995/03/08 15:24:12 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE EQDINC (X, N, XI, MAXNI, NI)


      INTEGER N,MAXNI,NI
      INTEGER K,MINNI,NK,I
      INTEGER ICEILX

      REAL DELX,DXK
      REAL X(N),XI(MAXNI)


      IF (MAXNI.LT.N)
     -  CALL HALT('EQDINC - Too few output elements')

* Check the input data
      DO 100 K=1,N-1
        IF (X(K+1).LT.X(K))
     -    CALL HALT('EQDINC - Abscissa values not in '//
     -              'increasing order')
 100  CONTINUE

* Special case, one input point
      IF (N.EQ.1) THEN
        XI(1)=X(1)
        NI=1

* Normal case
      ELSE
        MINNI=MAXNI-(N-1)
        DELX=(X(N)-X(1))/MINNI
        NI=0
        DO 140 K=1,N-1
          DXK=X(K+1)-X(K)
          NK=ICEILX(DXK/DELX)
          DO 120 I=0,NK-1
            NI=NI+1
            XI(NI)=I*(DXK/NK)+X(K)
 120      CONTINUE
 140    CONTINUE

        NI=NI+1
        IF (NI.LT.MINNI .OR. NI.GT.MAXNI)
     -    CALL HALT('EQDINC - Consistency check fails')
        XI(NI)=X(N)

      END IF


      RETURN

      END
