*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION PHSIIR (F, H, NSECT)
*
*
* Purpose:
*     Find the phase response for an cascade IIR filter
*
*
* Description:
*     This function calculates the phase response of an IIR filter
*     specified by its coefficients.  The filter is a cascade of second
*     order sections, with the z-transform of the i'th filter section
*     given as
*
*                h(1,i)*z**2 + h(2,i)*z + h(3,i)
*       H(i,z) = ------------------------------- .
*                   z**2 + h(4,i)*z + h(5,i)
*
*     If the given frequency corresponds to an uncancelled pole or zero
*     of the response, a warning message is printed and the phase value
*     is set to zero.
*
*
* Parameters:
* R <-  PHSIIR - Phase in radians.  This is the unwrapped (true) phase
*                of the cascaded sections.
*
* R ->  F      - Normalized frequency (normally in the range
*                0 to 0.5)
* R ->  H      - Array of groups of 5 filter coefficients, each
*                defining a biquadratic filter section
* I ->  NSECT  - Number of filter sections
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.7 $  $Date: 1995/03/08 15:21:40 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION PHSIIR (F, H, NSECT)


      REAL PI2
      PARAMETER (PI2=6.283 185 307)

      INTEGER NSECT
      INTEGER NZ,K

      REAL F
      REAL COSW,SINW,B2,B1,B0,C1,C0,AR,AI
      REAL H(5,*)


* Add the group delay of each second order section
      COSW = COS(PI2*F)
      SINW = SIN(PI2*F)
      NZ = 0
      PHSIIR = 0.0

      DO 100 K=1,NSECT

* The transfer function is
*
*                b2*z**2 + b1*z + b0   B(z)
*         H(z) = ------------------- = ----
*                  z**2 + c1*z + c0    C(z)
*
        B2 = H(1,K)
        B1 = H(2,K)
        B0 = H(3,K)
        C1 = H(4,K)
        C0 = H(5,K)

* The phase for H(w) is the phase for the numerator term
* less the phase for the denominator term.  On the unit
* circle express the numerator term as B(w) = R(w) + jX(w).
* The phase for the numerator term for w=2Pi f is
*
*     Phi(w) = arctan(X(w)/R(w))
*
* In the sequel, the numerator (and denominator) are expressed as
*
*               jw
*       B(w) = e   [ (b2+b0)cosw + b1 + j (b2-b0)sinw ]
*
* The exponential terms will cancel between the numerator and denominator.

* Phase due to numerator
        AR = B1 + (B2+B0)*COSW
        AI = (B2-B0)*SINW
        IF (AR.EQ.0.0 .AND. AI.EQ.0.0) THEN
          NZ = NZ+1
        ELSE
          PHSIIR = PHSIIR+ATAN2(AI,AR)
        END IF

* Phase due to denominator
        AR = C1 + (1.+C0)*COSW
        AI = (1.-C0)*SINW
        IF (AR.EQ.0.0 .AND. AI.EQ.0.0) THEN
          NZ = NZ-1
        ELSE
          PHSIIR = PHSIIR-ATAN2(AI,AR)
        END IF

 100  CONTINUE

      IF (NZ.NE.0) THEN
        CALL WARN('PHSIIR - Undefined phase')
        PHSIIR = 0.0
      END IF


      RETURN

      END
