*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION PHSCAS (F, H, NSECT)
*
*
* Purpose:
*     Find the phase response for an analog cascade filter
*
*
* Description:
*     This function calculates the phase response of a cascade analog
*     filter specified by its coefficients.  The filter is a cascade of
*     second order sections, with the Laplace transform of the i'th
*     filter section given as
*
*                h(1,i)*s**2 + h(2,i)*s + h(3,i)
*       H(i,s) = ------------------------------- .
*                   s**2 + h(4,i)*s + h(5,i)
*
*     If the given frequency corresponds to an uncancelled pole or zero
*     of the response, a warning message is printed and the phase value
*     is set to zero.
*
*
* Parameters:
* R <-  PHSCAS - Phase in radians.  This is the unwrapped (true) phase
*                of the cascaded sections.
*
* R ->  F      - Frequency
* R ->  H      - Array of groups of 5 filter coefficients, each
*                defining a biquadratic filter section
* I ->  NSECT  - Number of filter sections
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.6 $  $Date: 1995/03/08 15:21:34 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION PHSCAS (F, H, NSECT)


      REAL PI2
      PARAMETER (PI2=6.283 185 307)

      INTEGER NSECT
      INTEGER NZ,K

      REAL F
      REAL W,B2,B1,B0,C2,C1,C0,AR,AI
      REAL H(5,*)


* Add the group delay of each second order section
      W = PI2*F
      NZ = 0
      PHSCAS = 0.0

      DO 100 K=1,NSECT

* The transfer function is
*
*                b2*s**2 + b1*s + b0   B(s)
*         H(s) = ------------------- = ----
*                  s**2 + c1*s + c0    C(s)
*
        B2 = H(1,K)
        B1 = H(2,K)
        B0 = H(3,K)
        C2 = 1.
        C1 = H(4,K)
        C0 = H(5,K)

* The phase for H(jw) is the phase for the numerator term less
* the phase for the denominator term.  On the imaginary axis
* express the numerator term as B(jw) = R(w) + jX(w).  The
* phase for the numerator term for w=2Pi f is
*
*     Phi(w) = arctan(X(w)/R(w))
*
* In the sequel, the numerator (and denominator) are expressed as
*
*       B(jw) =  b0  - b2 w**2 + j b1 w
*

* Phase due to numerator
        AR = B0 - B2*W**2
        AI = B1*W
        IF (AR.EQ.0.0 .AND. AI.EQ.0.0) THEN
          NZ = NZ+1
        ELSE
          PHSCAS = PHSCAS+ATAN2(AI,AR)
        END IF

* Phase due to denominator
        AR = C0 - C2*W**2
        AI = C1*W
        IF (AR.EQ.0.0 .AND. AI.EQ.0.0) THEN
          NZ = NZ-1
        ELSE
          PHSCAS = PHSCAS-ATAN2(AI,AR)
        END IF

 100  CONTINUE

      IF (NZ.NE.0) THEN
        CALL WARN('PHSCAS - Undefined phase')
        PHSCAS = 0.0
      END IF


      RETURN

      END
