*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION GDELRP (H, NCOF, FLTTYP, SFREQ)
*
*
* Purpose:
*     Find the group response of a filter
*
*
* Description:
*     This routine calculates the group delay response of a filter
*     specified by its coefficients.
*
*     *** FIR or WIN:
*     For an FIR filter, the coefficients are the direct-form
*     coefficients, h(i)
*
*              N-1       -i
*       H(z) = SUM h(i) z    .
*              i=0
*
*     *** ALL:
*     For an all-pole IIR filter, the response is specified by the
*     direct-form feedback coefficients, h(i),
*
*               1                    N-1       -i
*       H(z) = ----  ,  where C(z) = SUM h(i) z    .
*              C(z)                  i=0
*
*     *** IIR:
*     For a general IIR filter, the response is specified in terms of
*     the coefficients of cascaded second order sections,
*
*              NSECT
*       H(z) = PROD H(k,z) .
*              k=1
*
*     Each second order section has a response defined by five
*     coefficients,
*
*               h(1,k)*z**2 + h(2,k)*z + h(3,k)
*      H(k,z) = -------------------------------  .
*                  z**2 + h(4,k)*z + h(k,5)
*
*     *** CAS:
*     For a cascade analog filter, the response is specified in terms
*     of the coefficients of cascaded second order sections,
*
*              NSECT
*       H(s) = PROD H(k,s) .
*              k=1
*
*     Each second order section has a response defined by five
*     coefficients,
*
*               h(1,k)*s**2 + h(2,k)*s + h(3,k)
*      H(k,s) = -------------------------------  .
*                  s**2 + h(4,k)*s + h(k,5)
*
*
*     The response is evaluated directly from the z-transform, with z
*     on the unit circle or from the Laplace transform with s on the
*     imaginary axis.
*
*
* Parameters:
* R <-  GDELRP - Output group delay value
*
* R ->  F      - Frequency at which the group delay is to be calculated
* R ->  H      - Array of filter coefficients.  For general IIR filters
*                and cascade analog filters, each group of 5
*                coefficients describes a second order section.
* I ->  NCOF   - Number of filter coefficients
* C ->  FLTTYP - Character string,
*                 ALL - all-pole filters
*                 CAS - cascade analog filters
*                 FIR - FIR filters
*                 IIR - IIR filters
*                 WIN - windows (same as FIR)
* R ->  SFREQ  - Sampling frequency
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.6 $  $Date: 1995/03/08 15:45:35 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION GDELRP (F, H, NCOF, FLTTYP, SFREQ)


      INTEGER NCOF
      INTEGER NSECT

      CHARACTER*(*) FLTTYP

      REAL F,SFREQ
      REAL H(0:NCOF-1)
      REAL GDLFIR,GDLIIR,GDLCAS


      IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN') THEN

* FIR filters
        GDELRP=GDLFIR(F/SFREQ,H,NCOF)/SFREQ

      ELSE IF (FLTTYP.EQ.'ALL') THEN

* All-pole filter
        GDELRP=-GDLFIR(F/SFREQ,H,NCOF)/SFREQ

      ELSE IF (FLTTYP.EQ.'IIR') THEN

* General IIR filters
        IF (MOD(NCOF,5).NE.0)
     -    CALL HALT('GDELRP - Invalid number of coefficients')
        NSECT=NCOF/5
        GDELRP=GDLIIR(F/SFREQ,H,NSECT)/SFREQ

      ELSE IF (FLTTYP.EQ.'CAS') THEN

* Cascade analog filters
        IF (MOD(NCOF,5).NE.0)
     -    CALL HALT('GDELRP - Invalid number of coefficients')
        NSECT=NCOF/5
        GDELRP=GDLCAS(F,H,NSECT)

      ELSE

        CALL HALT('GDELRP - Invalid filter type')

      END IF


      RETURN

      END
