*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION FMAGRP (F, H, NCOF, FLTTYP, SFREQ)
*
*
* Purpose:
*     Find the magnitude response of a filter
*
*
* Description:
*     This routine calculates the magnitude of the frequency response
*     of a filter specified by its coefficients.
*
*     *** FIR or WIN:
*     For an FIR filter, the coefficients are the direct-form
*     coefficients, h(i)
*
*              N-1       -i
*       H(z) = SUM h(i) z    .
*              i=0
*
*     *** ALL:
*     For an all-pole IIR filter, the response is specified by the
*     direct-form feedback coefficients, h(i),
*
*               1                    N-1       -i
*       H(z) = ----  ,  where C(z) = SUM h(i) z    .
*              C(z)                  i=0
*
*     Normally, the first coefficient is unity, i.e. h(0)=1.  A
*     non-unity value will result in a gain scaling of the response.
*
*     *** IIR:
*     For a general IIR filter, the response is specified in terms of
*     the coefficients of cascaded second order sections,
*
*              NSECT
*       H(z) = PROD H(k,z) .
*              k=1
*
*     Each second order section has a response defined by five
*     coefficients,
*
*               h(1,k)*z**2 + h(2,k)*z + h(3,k)
*      H(k,z) = -------------------------------  .
*                  z**2 + h(4,k)*z + h(k,5)
*
*     *** CAS:
*     For a cascade analog filter, the response is specified in terms
*     of the coefficients of cascaded second order sections,
*
*              NSECT
*       H(s) = PROD H(k,s) .
*              k=1
*
*     Each second order section has a response defined by five
*     coefficients,
*
*               h(1,k)*s**2 + h(2,k)*s + h(3,k)
*      H(k,s) = -------------------------------  .
*                  s**2 + h(4,k)*s + h(k,5)
*
*
*     The response is evaluated directly from the z-transform, with z
*     on the unit circle or from the Laplace transform with s on the
*     imaginary axis.
*
*
* Parameters:
* R <-  FMAGRP - Output amplitude value (non-negative)
*
* R ->  F      - Frequency at which the response is to be calculated
* R ->  H      - Array of filter coefficients.  For general IIR filters
*                and cascade analog filters, each group of 5
*                coefficients describes a second order section.
* I ->  NCOF   - Number of filter coefficients
* C ->  FLTTYP - Character string,
*                 ALL - all-pole filters
*                 CAS - cascade analog filters
*                 FIR - FIR filters
*                 IIR - IIR filters
*                 WIN - windows (same as FIR)
* R ->  SFREQ  - Sampling frequency
*
*
* Author / revision:
*     P. Kabal
*     $Revision: 1.6 $  $Date: 1995/03/08 15:45:29 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION FMAGRP (F, H, NCOF, FLTTYP, SFREQ)


      INTEGER NCOF
      INTEGER NSECT

      CHARACTER*(*) FLTTYP

      COMPLEX RSPCAS,RSPFIR,RSPIIR

      REAL F,SFREQ
      REAL H(0:NCOF-1)
      REAL YY


      IF (FLTTYP.EQ.'FIR' .OR. FLTTYP.EQ.'WIN') THEN

* FIR filters
        FMAGRP=CABS(RSPFIR(F/SFREQ,H,NCOF))

      ELSE IF (FLTTYP.EQ.'ALL') THEN

* All-pole filter
        YY=CABS(RSPFIR(F/SFREQ,H,NCOF))
        IF (YY.EQ.0.0) THEN
          CALL WARN('FMAGRP - Fixup for unit circle pole')
          FMAGRP=1E20
        ELSE
          FMAGRP=1./YY
        END IF

      ELSE IF (FLTTYP.EQ.'IIR') THEN

* General IIR filters
        IF (MOD(NCOF,5).NE.0)
     -    CALL HALT('FMAGRP - Invalid number of coefficients')
        NSECT=NCOF/5
        FMAGRP=CABS(RSPIIR(F/SFREQ,H,NSECT))

      ELSE IF (FLTTYP.EQ.'CAS') THEN

* Cascade analog filters
        IF (MOD(NCOF,5).NE.0)
     -    CALL HALT('FMAGRP - Invalid number of coefficients')
        NSECT=NCOF/5
        FMAGRP=CABS(RSPCAS(F,H,NSECT))

      ELSE

        CALL HALT('FMAGRP - Invalid filter type')

      END IF


      RETURN

      END
