/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void SPconvMSpoly (const float x[], double a, float z[], int N)

Purpose:
  Convolve coefficients for monic symmetric polynomials

Description:
  This routine convolves two sets of coefficients to form an output coefficient
  array.  If the coefficients are considered to be polynomial coefficients,
  this operation is equivalent to polynomial multiplication.

  The input coefficient array contains the coefficients for a monic (leading
  coefficient equal to one), symmetric polynomial.  This polynomial has an odd
  number of terms,
    X(D) = 1 + x(1) D + x(2) D^2 + ... + x(N) D^N
             + x(N-1) D^(N+1) + ... + x(1) D^(2N-1) + D^(2N)
   The polynomial is specified by only N coefficients, x(1),...,x(N).  The
   second input polynomial is
     Y(D) = 1 + a D + D^2 ,
   which is also a monic symmetric polynomial with an odd number of
   coefficients.  It is specified by the single coefficient a.  The product
   of X(D) and Y(D) has 2N+3 coefficients.  However, the product is a monic,
   symmetric polynomial and hence can be specified by N+1 terms.

Parameters:
   -> const float x[]
      Input coefficient array.  This array contains N coefficients specifying a
      monic, symmetric polynomial with a total of 2N+1 terms.
   -> double  a
      Coefficient of the second input polynomial.  This value is the middle
      coefficient for a three term monic, symmetric polynomial.
  <-  float z[]
      This array contains the N+1 coefficients resulting from the convolution
      of the input polynomial coefficients with those for a three term monic,
      symmetric polynomial (specified by the coefficient a).  The order of
      operations is such that this array can share storage with the array x.
   -> int  N
      Number of input coefficients

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/05/06 19:24:37 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: SPconvMSpoly.c 1.10 1996/05/06 libtsp-V2R7a $";

#include <libtsp/nucleus.h>

/* 1-offset indexing */
#define X(i)		x[i-1]
#define Z(i)		z[i-1]

void
SPconvMSpoly (x, a, z, N)

     const float x[];
     double a;
     float z[];
     int N;

{
  int k;
  float A;

/*
   Consider a monic, symmmetric polynomial with an odd number of coefficients
   (2N+1).  This polynomial can be specified by N unique coefficients,
   x(1),...,x(N).  Let
     X(D) = x(0) + x(1) D + x(2) D^2 + ... + x(2N) D^(2N)
   Then if this polynomial is monic and symmetric,
     x(0) = x(2N) = 1  (monic and symmetric)
     x(1) = x(2N-1)
      ...    ...
     x(n) = x(2N-n)    (general term)
      ...    ...
     x(N-1) = x(N+1)
     x(N)              (not paired)
   Consider another monic, symmetric polynomial with 3 coefficients,
     Y(D) = 1 + a D + D^2
   The convolution of the coefficients of X(D) and Y(D) gives another monic,
   symmetric polynomial with an odd number of coefficients.  Let
   Z(D) = Y(D) X(D), then
     z(0) = x(0) = 1
     z(1) = x(1) + ax(0) = x(1) + a
     z(2) = x(2) + ax(1) + x(0) = x(2) + ax(1) + 1
     z(3) = x(3) + ax(2) + x(1)
     ...     ...
     z(n) = x(n) + ax(n-1) + x(n-2)  ... general term
     ...     ...
     z(N-1) = x(N-1) + ax(N-2) + x(N-3)
     z(N) = x(N) + ax(N-1) + x(N-2)
     z(N+1) = x(N+1) + ax(N) + x(N-1) = 2x(N+1) + ax(N)

   Notes:
    1) z(0) need not be calculated
    2) terms z(N+2),...,z(2N+2) need not be calculated since they can be
       obtained by symmetry
    3) terms z(3),...z(N) are of the same form as the general term
    4) term z(N+1) uses the symmetry of X(D), x(N+1)=x(N-1)
*/

/*
   Convolve the coefficients by summing shifted versions of X(.) weighted by a.
   By choosing the order of the operations appropriately, the result can
   overlay the input array X(.).
*/

  A = a;

  if (N >= 2) {
    Z(N+1) = (X(N-1) + X(N-1)) + A * X(N);
    for (k = N; k >= 3; --k) {
      Z(k) = X(k) + A * X(k-1) + X(k-2);
    }
    Z(2) = X(2) + A * X(1) + 1.0;
    Z(1) = X(1) + A;
  }
  else if (N == 1) {
    Z(2) = A * X(1) + 2.0;
    Z(1) = X(1) + A;
  }
  else if (N == 0) {
    Z(1) = A;
  }

  return;
}
