/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void SPlsfXpc (const float lsf[], float pc[], int Np)

Purpose:
  Convert LSF's to predictor coefficients

Description:
  This routine converts a vector of line spectral frequencies to the
  corresponding vector of predictor coefficients.  The LSF's are the
  frequencies (angles) corresponding to roots on the unit circle.  These roots
  are the roots of two polynomials, one symmetric, the other anti-symmetric.
  The odd-numbered LSF's specify roots of one polynomial, while the even-
  numbered LSF's specify the roots of the other polynomial.  The polynomials
  are formed by the polynomial multiplication of the factors representing
  conjugate pairs of roots.  The predictor coefficients are formed by the
  addition of the two polynomials.

  Line spectral frequencies and predictor coefficients are usually expressed
  algebraically as vectors with one-offset indexing.  The correspondence to the
  zero-offset C-arrays is as follows.
    l(1) <==> lsf[0]       first (lowest frequency) line spectral frequency
    l(i) <==> lsf[i-1]     1 <= i < Np
    p(1) <==> pc[0]        predictor coefficient corresponding to lag 1
    p(i) <==> pc[i-1]      1 <= i < Np

Parameters:
   -> const float lsf[]
      Array of Np line spectral frequencies.  Each line spectral frequency lies
      in the range 0 to pi.  If the LSF's are well-ordered (ascending order),
      the resulting predictor corresponds to a minimum phase prediction error
      filter.
   -> int Np
      Number of coefficients (at most 50)
  <-  float pc[]
      Output array of Np predictor coefficients.  These are the coefficients
      of the predictor filter, with pc[0] being the predictor coefficient
      corresponding to lag 1, and pc[Np-1] corresponding to lag Np.
   -> int Np
      Number of coefficients (at most 50)

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.15 $  $Date: 1996/05/06 18:30:25 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: SPlsfXpc.c 1.15 1996/05/06 libtsp-V2R7a $";

#include <math.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>

#define MAXNP		50

/* one-offset indexing */
#define PC(i)		pc[(i)-1]
#define LSF(i)		lsf[(i)-1]

void
SPlsfXpc (lsf, pc, Np)

     const float lsf[];
     float pc[];
     int Np;

{
  float fa[(MAXNP+1)/2+1];
  float fb[MAXNP/2+1];
  double a;
  int i, k, Nc, M;

  fa[0] = 1.0;
  fb[0] = 1.0;

/*
   Each line spectral frequency w contributes a second order polynomial of the
   form Y(D)=1-2*cos(w(i))*D+D^2.  These polynomials are formed for each line
   spectral frequency and then multiplied together.  Alternate line spectral
   frequencies are used to form two polynomials with interlacing roots on the
   unit circle.  These two polynomials are multiplied by 1+D and 1-D if Np is
   even or by 1 and 1-D^2 if Np is odd.  This gives the symmetric and anti-
   symmetric polynomials that are added to give the predictor coefficients.
*/
  if (Np > MAXNP)
    UThalt ("SPlsfXpc: Too many coefficients");

/* Form a symmetric Fa(D) by multiplying together second order polynomials */
  Nc = 0;
  for (i = 1; i <= Np; i += 2) {
    a = -2.0 * cos (LSF(i));
    SPconvMSpoly (&fa[1], a, &fa[1], Nc);
    ++Nc;
  }

/* Form a symmetric Fb(D) by multiplying together second order polynomials */
  Nc = 0;
  for (i = 2; i <= Np; i += 2) {
    a = -2.0 * cos (LSF(i));
    SPconvMSpoly  (&fb[1], a, &fb[1], Nc);
    ++Nc;
  }

/*
   Both Fa(D) and Fb(D) are symmetric, with leading coefficient equal to unity.
   Exclusive of the leading coefficient, the number of coefficients needed to
   specify Fa(D) and Fb(D) is:
     Np      Fa(D)     Fb(D)
    even     Np/2      Np/2
    odd    (Np+1)/2  (Np-1)/2
*/

  if (Np % 2 != 0) {

/* ----- Np odd */
/* Fb(D) is multiplied by the factor (1-D^2) */
    M = (Np - 1) / 2;
    for (i = M; i >= 2; --i) {
      fb[i] = fb[i] - fb[i-2];
    }

/*
   Form the predictor filter coefficients.  Fa(D) is symmetric and Fb(D) is
   anti-symmetric.  Since only the first half of the coefficients are available
   symmetries are used to get the other half.
*/
    for (i = 1, k = Np; i <= M; ++i, --k) {
      PC(i) = -0.5 * (fa[i] + fb[i]);
      PC(k) = -0.5 * (fa[i] - fb[i]);
    }
    PC(i) = -0.5 * fa[i];
  }

  else {

/* ----- Np even */
/* Fa(D) is multiplied by the factor (1+D) */
/* Fb(D) is multiplied by the factor (1-D) */
    M = Np / 2;
    for (i = M; i >= 1; --i) {
      fa[i] = fa[i] + fa[i-1];
      fb[i] = fb[i] - fb[i-1];
    }

/* Form the predictor filter coefficients */
/* Fa(D) is symmetric and Fb(D) is anti-symmetric. */
    for (i = 1, k = Np; i <= M; ++i, --k) {
      PC(i) = -0.5 * (fa[i] + fb[i]);
      PC(k) = -0.5 * (fa[i] - fb[i]);
    }
  }
  return;
}
