/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double SPcovCXpc (const float *Cov[], float pc[], int Np)

Purpose:
  Find predictor coefficients using the covariance lattice method (Cumani)

Description:
  This routine calculates the parameters for a lattice predictor which
  minimizes the sum of the forward and backward errors.  The reflection
  coefficients which describe the lattice filter are converted to the predictor
  coefficients which are returned.  This procedure guarantees a minimum phase
  prediction error filter.

  Consider a linear predictor with Np coefficients,
            Np
    y(k) = SUM p(i) x(k-i) ,
           i=1
  where x(i) is the input signal.  This routine returns the predictor
  cofficients.

  This procedure is based on Makhoul's covariance lattice method as modified
  by Cumani to compute the result using reflection coefficients.  The
  reflection coefficients are chosen to minimize a weighted sum of the
  forward and backward prediction errors.

  The original covariance lattice method of Makhoul allowed for minimizing a
  weighted combination of the forward and backward prediction errors.  Here the
  forward and backward prediction errors have equal weight.  Furthermore, the
  computation of the reflection coefficients is modified to as specified in the
  EIA/TIA IS-54-B digital mobile cellular radio standard.

  The computation uses a correlation matrix Cov with elements defined as

    Cov(i,j) = E[x(k-i) x(k-j)],  for 0 <= i,j <= Np.

  The expectation operator E[.] is often replaced by a sum over k over a finite
  interval.  Minimization of the prediction error over this interval defines
  the so-called covariance method for determining the linear prediction
  coefficients.

  References:
  1: J. Makhoul, "Stable and efficient lattice methods for linear prediction",
     IEEE Trans. Acoustics, Speech, Signal Processing, vol. ASSP-25, pp.
     423-428, October 1977.
  2: A. Cumani, "On a covariance-lattice algorithm for linear prediction",
     Proc. IEEE Conf. Acoustics, Speech, Signal Processing, Paris, pp. 651-654,
     May 1982.
  3: EIA/TIA Interim Standard IS-54-B, "Cellular System Dual-Mode Mobile
     Station - Base Station Compatibility Standard", Telecommunications
     Industry Association, April 1992.

  Predictor coefficients are usually expressed algebraically as vectors with
  1-offset indexing.  The correspondence to the 0-offset C-arrays is as
  follows.
    p(1) <==> pc[0]       predictor coefficient corresponding to lag 1
    p(i) <==> pc[i-1]     1 <= i < Np

Parameters:
  <-  double SPcovCXpc
      Resultant prediction error energy
   -> const float *Cov[]
      Cov is an array of pointers to the rows of an Np+1 by Np+1 symmetric
      positive definite correlation matrix.  Only the lower triangular portion
      of Cov is accessed.  Note that with ANSI C, if the actual parameter
      is not declared to have the const attribute, an explicit cast to
      (const float **) is required.
  <-  const float pc[]
      Np element vector of predictor coefficients.  Coefficient pc[i] is the
      predictor coefficient corresponding to lag i+1.
   -> int Np
      Number of prediction coefficients

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.6 $  $Date: 1996/05/31 12:40:20 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: SPcovCXpc.c 1.6 1996/05/31 libtsp-V2R7a $";

#include <libtsp.h>

double
SPcovCXpc (Cov, pc, Np)

     const float *Cov[];
     float pc[];
     int Np;

{
  int i, j, m;
  float rm;
  float **F, **B, **C;
  float *rc;
  float Ferr, Berr, Cerr;
  double denom;

/* Allocate space for the matrices */
  F = (float **) MAfAllocMat (3*Np, Np);
  B = F + Np;
  C = B + Np;
  rc = (float *) UTmalloc (Np * sizeof (float));

/* Initialize the matrices */
  for (i = 0; i < Np; ++i) {
    for (j = 0; j < i; ++j) {
      F[i][j] = Cov[i][j];
      B[i][j] = Cov[i+1][j+1];
      C[i][j] = Cov[i][j+1];
      C[j][i] = Cov[i+1][j];	/* = Cov[j][i+1] */
    }
    F[i][i] = Cov[i][i];
    B[i][i] = Cov[i+1][i+1];
    C[i][i] = Cov[i+1][i];
  }

  for (m = 1; m <= Np; ++m) {

/* Set the new reflection coefficient as the ratio of the cross-error to the
   weighted sum of the forward and backward errors.  This formula can be
   expressed as
     k(m) = -Cerr / (g * Berr + (1-g) * Ferr).
   Here g=0.5.  This equal weighting guarantees that the prediction error
   filter is minimum phase.

   The original covariance lattice formulation (Makhoul) uses
     Cerr = C[0][0],  Berr = B[0][0], Ferr = F[0][0].
   The formula is modified in IS-54-B to become
     Cerr = 0.5 * (C[0][0] + C[Np-m][Np-m]),
     Berr = 0.5 * (B[0][0] + B[Np-m][Np-m]),
     Ferr = 0.5 * (F[0][0] + F[Np-m][Np-m]).

   If both the forward error and the backward error are zero, then
   (theoretically) the cross-error should be zero.  With zero error, the
   next reflection coefficient should be chosen to be zero.
*/
    Ferr = F[0][0] + F[Np-m][Np-m];
    Berr = B[0][0] + B[Np-m][Np-m];
    Cerr = C[0][0] + C[Np-m][Np-m];
    denom = 0.5 * (Berr + Ferr);
    if (denom != 0.0)
      rm = -Cerr / denom;
    else
      rm = 0.0;
    rc[m-1] = rm;

/* Update the error correlations
   The updates are arranged so as to allow the update to be done in-place.
   Note that F[i][j] and B[i][j] are symmetric - only the lower triangular
   part is used.  C[i][j] is not symmetrical.
*/
    for (i = 0; i < Np-m; ++i) {
      for (j = 0; j < i; ++j) {
        F[i][j] = F[i][j] + rm * (C[i][j] + C[j][i] + rm * B[i][j]);
        B[i][j] = B[i+1][j+1] + rm * (C[i+1][j+1] + C[j+1][i+1]
                                      + rm * F[i+1][j+1]);
        C[i][j] = C[i][j+1] + rm * (F[i][j+1] + B[i][j+1] + rm * C[j+1][i]);
        C[j][i] = C[j][i+1] + rm * (F[i+1][j] + B[i+1][j] + rm * C[i+1][j]);
      }
      F[i][i] = F[i][i] + rm * (C[i][i] + C[i][i] + rm * B[i][i]);
      B[i][i] = B[i+1][i+1] + rm * (C[i+1][i+1] + C[i+1][i+1]
                                    + rm * F[i+1][i+1]);
      C[i][i] = C[i][i+1] + rm * (F[i+1][i] + B[i+1][i] + rm * C[i+1][i]);
    }
  }

/* Last update for F[i][j] to get the forward error */
  F[0][0] = F[0][0] + rm * (C[0][0] + C[0][0] + rm * B[0][0]);

/* Convert to predictor coefficients */
  SPrcXpc (rc, pc, Np);

  MAfFreeMat (F);
  UTfree ((void *) rc);

  return F[0][0];
}
