/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MSintLin (const float x[], const float y[], int N, const float xi[],
		float yi[], int Ni)

Purpose:
  Interpolate a table of values using a piecewise linear interpolant

Description:
  This routine calculates interpolated values for a function defined by a table
  of reference data values.  The interpolated values are found by passing a
  piecewise linear interpolant through the reference data values.

Parameters:
   -> const float x[]
      Abscissa values for the reference points.  These values must be in
      increasing order.
   -> const float y[]
      Ordinate values for the reference points
   -> int N
      Number of reference points
   -> const float xi[]
      Abscissa values for the interpolated points.  These values must be
      bounded by x[0] and x[N-1].  The values xi[i] need not be in any
      particular order.
  <-  float yi[]
      Resulting interpolated ordinate values
   -> int Ni
      Number of interpolated values

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.6 $  $Date: 1996/05/06 18:25:39 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MSintLin.c 1.6 1996/05/06 libtsp-V2R7a $";

#include <libtsp.h>
#include <libtsp/nucleus.h>

void
MSintLin (x, y, N, xi, yi, Ni)

     const float x[];
     const float y[];
     int N;
     const float xi[];
     float yi[];
     int Ni;

{
  int i, k;
  float dx, a, b;

/* Check the reference abscissa values */
  for (k = 1; k < N; ++k) {
    if (x[k] < x[k-1])
      UThalt ("MSintLin: Abscissa values not in increasing order");
  }

  /* Special case: only one reference point */
  if (N == 1) {
    for (i = 0; i < Ni; ++i) {
      if (xi[i] != x[0])
	UThalt ("MSintLin: Abscissa value %g not in range [%g, %g]",
		xi[i], x[0], x[0]);
      yi[i] = y[0];
    }
  }

  /* General case: 2 or more reference points */
  /* N reference points -> N-1 intervals */
  else {
    for (i = 0; i < Ni; ++i) {
      /* Search for the bracketing interval */
      k = SPquant (xi[i], x, N + 1);		/* N+1 quantizer regions */
      if (k == 0 && xi[i] == x[0])
	k = 1;
      if (k <= 0 || k >= N)			/* reject 2 outer regions */
	UThalt ("MSintLin: Abscissa value %g not in range [%g, %g]",
		xi[i], x[0], x[N-1]);

/* Interval found: x[k-1] <= xi[i] <= x[k], with 1 <= k <= N-1 */
      dx = x[k] - x[k-1];
      a = (xi[i] - x[k-1]) / dx;
      b = (x[k] - xi[i]) / dx;
      yi[i] = b * y[k-1]  + a * y[k];
    }
  }
  return;
}
