/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFopenWrite (const char Fname[], int Fformat, long int Nchan,
                      double Sfreq, FILE *fpout)

Purpose:
  Open an audio file for writing

Description:
  This routine opens an audio file for writing.  This routine sets up the audio
  file parameters to write data of the given format to the audio file.  After
  writing data to the file, the routine AFclose should be called to update the
  file header information and close the file.

  This routine can write AFsp (Sun) audio files, RIFF WAVE files and
  headerless audio files.

  AFsp audio file:
    The AFsp audio file format uses a file header which is compatible with a
    Sun audio file header.  By default, information consisting of the date, the
    user and the program creating the file is written to the header.  The
    routine AFsetHInfo can be called before calling this routine to specify
    additional information to be written to the header.  Any such additional
    header information is reset to null by this routine.  8-bit mu-law, 8-bit
    A-law, 8-bit integer, 16-bit integer, and 32-bit IEEE floating-point data
    formats are supported.
  RIFF WAVE file:
    8-bit mu-law, 8-bit A-law, 8-bit integer, and 16-bit integer data formats
    are supported.
  AIFF-C file:
    8-bit mu-law, 8-bit A-law, 8-bit integer, and 16-bit integer data formats
    are supported.
  Headerless file:
    8-bit mu-law, 8-bit A-law, offset-binary 8-bit integer, 8-bit integer,
    16-bit integer, 32-bit floating-point, and text data formats are supported.
    A text format file has the data in character form, one value to a line.

  For the fixed point file data representations, input values in the following
  ranges will be converted without clipping.
     data format       allowed values
    8-bit mu-law   - [ -32636, +32636 ]
    8-bit A-law    - [ -32768, +32768 ]
    8-bit integer  - [ -16384, -16383 ]
    16-bit integer - [ -32768, +32767 ]
  For AFsp files containing floating-point data, values are scaled by
  1/32768 before being written to the file.  For data values between
  [ -32768, +32768 ], the file data will lie in the range [ -1, +1].  For
  headerless files of any data format, the values are written scaled by unity.

Parameters:
  <-  AFILE *AFopenWrite
      Audio file pointer for the audio file
   -> const char Fname[]
      Character string specifying the file name
   -> int Fformat
      Audio file format code, evaluated as the sum of a data format code and a
      file type,
        Fformat = Dformat + Ftype
      For Ftype equal to zero, the file will have the standard AFsp audio file
      header.  The Ftype flag allows for other file types as described below.
      Note that not all data formats are allowed for all file types.
      Dformat: data format
        FD_MULAW8  = 1,  mu-law 8-bit data
        FD_ALAW8   = 2,  A-law 8-bit data
        FD_UINT8   = 3,  offset-binary 8-bit integer data
        FD_INT8    = 4,  two's-complement 8-bit integer data
        FD_INT16   = 5,  two's-complement 16-bit integer data
        FD_FLOAT32 = 6,  32-bit floating-point data
        FD_TEXT    = 7,  text data
      Ftype: file type
        FW_AFsp = FW_SUN     = 0, AFsp (Sun) audio file
        FW_WAVE              = 256, RIFF WAVE file
        FW_AIFF_C            = 512, AIFF-C audio file
        FW_NH_EB             = 768, Headerless file (big-endian byte order)
        FW_NH_EL             = 1024, Headerless file (little-endian byte order)
        FW_NH = FW_NH_NATIVE = 1280, Headerless file (native byte order)
        FW_NH_SWAP           = 1536, Headerless file (swapped byte order)
   -> long int Nchan
      Number of channels
   -> double Sfreq
      Sampling frequency
   -> FILE *fpout
      File pointer for printing audio file information.  If fpout is not NULL,
      information about the audio file is printed on the stream selected by
      fpout.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.35 $  $Date: 1996/08/14 18:10:28 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFopenWrite.c 1.35 1996/08/14 libtsp-V2R7a $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

AFILE *
AFopenWrite (Fname, Fformat, Nchan, Sfreq, fpout)

     const char Fname[];
     int Fformat;
     long int Nchan;
     double Sfreq;
     FILE *fpout;

{
  AFILE *AFp;
  int Dformat, Ftype;

/* Set up the audio file parameters and write the header information */
  Dformat = Fformat % FW_MOD;
  Ftype = Fformat - Dformat;
  switch (Ftype) {
  case FW_AFSP:
    AFp = AFsetAUpar (Fname, Dformat, Nchan, Sfreq, fpout);
    break;
  case FW_WAVE:
    AFp = AFsetWVpar (Fname, Dformat, Nchan, Sfreq, fpout);
    break;
  case FW_AIFF_C:
    AFp = AFsetAIpar (Fname, Dformat, Nchan, Sfreq, fpout);
    break;
  case FW_NH_EB:
  case FW_NH_EL:
  case FW_NH_NATIVE:
  case FW_NH_SWAP:
    AFp = AFsetNHpar (Fname, Fformat, Nchan, Sfreq, fpout);
    break;
  default:
    UThalt ("AFopenWrite: Invalid file type");
    break;
  }

/* Reset the user supplied header info */
  AFresHinfo ();

  return AFp;
}
