*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE WRTFIR (NAMEFL, ITYPE, H, NTAP, IBAND, NBANDS,
*                        FREQ, VAL, WEIGHT, VLIML, VLIMU)
*
*
* Purpose:
*     Write an FIR filter coefficient file
*
*
* Description:
*     This subroutine writes a coefficient file for an FIR filter.  The
*     coefficient file is created and written to as unit number 3.  The
*     data written to the file includes a header containing the design
*     specification.  This is followed by the filter coefficients.  The
*     file is then closed.  The file name is also printed.
*
*
* Parameters:
* C ->  NAMEFL - File name for the coefficient file.  If the name is
*                all blank, no file is written.
* I ->  ITYPE  - Filter type coded as follows.
*                1 - Multiple passband/stopband filter
*                2 - Multiple passband/stopband filter
*                    (sin(x)/x compensation)
*                3 - Differentiator
*                4 - Hilbert transform filter
* R ->  H      - Array of coefficients (NTAP values)
* I ->  NTAP   - Number of filter coefficients
* I ->  IBAND  - Frequency band specification index.  The array FREQ
*                specifies the frequency bands.
*                 Band    Lower Band Edge          Upper Band Edge
*                  1     FREQ(1)                  FREQ(IBAND(1))
*                  2     FREQ(IBAND(1)+1)         FREQ(IBAND(2))
*                 ...        ...                    ...
*                NBANDS  FREQ(IBAND(NBANDS-1)+1)  FREQ(IBAND(NBANDS))
*                VAL(i) and WEIGHT(i) specify the desired value and
*                weight at frequency FREQ(i).
* I ->  NBANDS - Number of bands specified
* R ->  FREQ   - Array of normalized frequencies.  These are in
*                increasing order.
* R ->  VAL    - Array of desired amplitude values (desired slopes for
*                differentiators)
* R ->  WEIGHT - Array of desired weightings.  For differentiators,
*                the given weighting applies to the slope.
* R ->  VLIML  - Array of lower constraints. If the constraint takes on
*                the value -1E20, it is assumed not to have been
*                specified.
* R ->  VLIMU  - Array of upper constraints. If the constraint takes on
*                the value +1E20, it is assumed not to have been
*                specified.
*
*
* Routines required:
*     LENSTR - Find the length of a whitespace terminated string
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:38:27 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE WRTFIR (NAMEFL, ITYPE, H, NTAP, IBAND, NBANDS,
     -                   FREQ, VAL, WEIGHT, VLIML, VLIMU)


      INTEGER NKEY,BPF,REC,DIF,HIL,ICFF
      PARAMETER (NKEY=4,BPF=1,REC=2,DIF=3,HIL=4)
      PARAMETER (ICFF=3)

      CHARACTER*(*) NAMEFL,FULLNM*255,FLTTYP(NKEY)*39
      CHARACTER*8 CSTR

      LOGICAL LIMS

      INTEGER ITYPE,NTAP,IBAND(*),NBANDS
      INTEGER NC,IU,K,IL,I,NCHR
      INTEGER LENSTR

      REAL H(0:NTAP-1),FREQ(*),VAL(*),WEIGHT(*),VLIML(*),VLIMU(*)

      DATA FLTTYP/'Bandpass Filter',
     -            'Bandpass Filter (sin(x)/x compensation)',
     -            'Differentiator',
     -            'Hilbert Transform Filter'/


      NC=LENSTR(NAMEFL)
      IF (NC.GT.0) THEN
        OPEN (UNIT=ICFF,FILE=NAMEFL(1:NC),STATUS='NEW')
        INQUIRE (UNIT=ICFF,NAME=FULLNM)

        WRITE (UNIT=ICFF,FMT=3000) FLTTYP(ITYPE)

        WRITE (UNIT=ICFF,FMT=3100)
        IU=0
        DO 120 K=1,NBANDS
          IL=IU+1
          IU=IBAND(K)
          WRITE (UNIT=CSTR,FMT=4000) K
          DO 100 I=IL,IU
            LIMS=VLIML(I).NE.-1E20 .OR. VLIMU(I).NE.+1E20
            IF (LIMS) THEN
              WRITE (UNIT=ICFF,FMT=3200)
     -              CSTR,FREQ(I),VAL(I),WEIGHT(I),VLIML(I),VLIMU(I)
            ELSE
              WRITE (UNIT=ICFF,FMT=3300)
     -              CSTR,FREQ(I),VAL(I),WEIGHT(I)
            END IF
            CSTR=' '
 100      CONTINUE
 120    CONTINUE
        WRITE (UNIT=ICFF,FMT=3400) (H(I),I=0,NTAP-1)

        CLOSE (UNIT=ICFF)
        NCHR=LENSTR(FULLNM)
        WRITE (UNIT=*,FMT=2000) FULLNM(1:NCHR)
      END IF


      RETURN

 2000 FORMAT (/' Coefficient file: ',A)
 3000 FORMAT ('!FIR - ',A)
 3100 FORMAT ('!         Frequency  Value       Weight    ',
     -                  'Lower Limit Upper Limit')
 3200 FORMAT ('!',A,F9.5,1PG12.4,1PG12.4,1PG12.4,1PG12.4)
 3300 FORMAT ('!',A,F9.5,1PG12.4,1PG12.4,'    ----    ',
     -                                   '    ----    ')
 3400 FORMAT (1PE15.6,1PE15.6,1PE15.6,1PE15.6,1PE15.6)
 4000 FORMAT ('Band',I3,':')

      END
