*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE TRSPEC (ITYPE, NGRID, FGRID, XGRID,
*                        DES, DEST, WT, WTT,
*                        DLIML, DLIMLT, DLIMU, DLIMUT)
*
*
* Purpose:
*     Transform the FIR design problem into a canonical form
*
*
* Description:
*     Given the specifications of a filter on a frequency grid, this
*     routine modifies the specifications for the design of different
*     types of filters.
*
*     There are four cases to be considered for the linear phase
*     filter,
*              N-1       -n
*       H(z) = SUM h(n) z  .
*              n=0
*
*      Case     symmetry      even/odd
*       1)    h(n)=h(N-1-n)    N odd     bandpass filter
*       2)    h(n)=h(N-1-n)    N even    bandpass filter
*       3)    h(n)=-h(N-1-n)   N odd     differentiator/Hilbert tr.
*       4)    h(n)=-h(N-1-n)   N even    differentiator/Hilbert tr.
*
*     The frequency response, H(w) = H(exp(jw)), can be written in the
*     following form for each of the cases,
*                                   M-1
*        H(w) = exp(jw(N-1)/2) Q(w) SUM a(n) cos(wn) .
*                                   n=0
*
*      Case     Q(w)       No. coef. (M)
*       1)        1          (N-1)/2+1
*       2)    2 cos(w/2)      N/2
*       3)    2j sin(w)      (N-3)/2+1
*       4)    2j sin(w/2)     N/2
*
*     The given specifications will be converted to an equivalent
*     specification for approximation by a sum of cosines.  The phase
*     factors in H(w) do not enter into the magnitude approximation.
*     The fixed factor Q(w) is absorbed into the desired value,
*     constraint values, and weighting.
*
*
* Parameters:
* I ->  ICASE  - Filter case.
*                1 - odd number of coefficients, bandpass filter
*                2 - even number of coefficients, bandpass filter
*                3 - odd number of coefficients, differentiator or
*                    Hilbert transform
*                4 - even number of coefficients, differentiator or
*                    Hilbert transform
* I ->  NGRID  - Number of elements in each of FGRID, XGRID, DES, WT,
*                DLIML, and DLIMU
* R ->  FGRID  - Normalized frequency values on the grid in the range
*                [0,1/2].  These values must be in strictly increasing
*                order and result in distinct values of XGRID.  For
*                ICASE equal to 3 or 4, FGRID(i) must not be zero.
*                For ICASE equal to 2 or 3, FGRID(i) must not be equal
*                to 1/2.
* R <-  XGRID  - Abscissa values, cosines of the frequency values.
*                XGRID(i) is calculated as COS(2*Pi*FGRID(i)).
* R ->  DES    - Desired values on the grid
* R <-  DEST   - Transformed desired values on the grid
* R ->  WT     - Weight values on the grid
* R <-  WTT    - Transformed weight values on the grid
* R ->  DLIML  - Lower constraint values on the grid
* R <-  DLIMLT - Transformed lower constraint values on the grid
* R ->  DLIMU  - Upper constraint values on the grid
* R <-  DLIMUT - Transformed upper constraint values on the grid
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:36:33 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE TRSPEC (ICASE, NGRID, FGRID, XGRID,
     -                   DES, DEST, WT, WTT,
     -                   DLIML, DLIMLT, DLIMU, DLIMUT)


      REAL PI,PI2
      PARAMETER (PI=3.141 592 654, PI2=6.283 185 307)

      INTEGER ICASE,NGRID
      INTEGER I

      REAL FGRID(NGRID),XGRID(NGRID),DES(NGRID),DEST(NGRID),
     -     WT(NGRID),WTT(NGRID),DLIML(NGRID),DLIMLT(NGRID),
     -     DLIMU(NGRID),DLIMUT(NGRID)
      REAL CH


* Set up a new approximation problem which is equivalent
* to the original problem

* Bandpass filter (odd number of coefficients)
* no change
      IF (ICASE.EQ.1) THEN
        DO 200 I=1,NGRID
          DEST(I)=DES(I)
          WTT(I)=WT(I)
          DLIMLT(I)=DLIML(I)
          DLIMUT(I)=DLIMU(I)
 200    CONTINUE

* Bandpass filter (even number of coefficients)
      ELSE IF (ICASE.EQ.2) THEN
        DO 220 I=1,NGRID
          CH=2.*COS(PI*FGRID(I))
          DEST(I)=DES(I)/CH
          WTT(I)=CH*WT(I)
          DLIMLT(I)=DLIML(I)/CH
          DLIMUT(I)=DLIMU(I)/CH
 220    CONTINUE

* Differentiator or Hilbert transform (odd number of coefficients)
      ELSE IF (ICASE.EQ.3) THEN
        DO 240 I=1,NGRID
          CH=2.*SIN(PI2*FGRID(I))
          DEST(I)=DES(I)/CH
          WTT(I)=CH*WT(I)
          DLIMLT(I)=DLIML(I)/CH
          DLIMUT(I)=DLIMU(I)/CH
 240    CONTINUE

* Differentiator or Hilbert transform (even number of coefficients)
      ELSE IF (ICASE.EQ.4) THEN
        DO 260 I=1,NGRID
          CH=2.*SIN(PI*FGRID(I))
          DEST(I)=DES(I)/CH
          WTT(I)=CH*WT(I)
          DLIMLT(I)=DLIML(I)/CH
          DLIMUT(I)=DLIMU(I)/CH
 260    CONTINUE

      END IF


* Change the frequency axis to cosine values
      DO 300 I=1,NGRID
        XGRID(I)=COS(PI2*FGRID(I))
 300  CONTINUE
      DO 320 I=2,NGRID
        IF (FGRID(I).LE.FGRID(I-1))
     -    CALL HALT('TRSPEC - Frequency values not in '//
     -              'increasing order')
        IF (XGRID(I).GE.XGRID(I-1))
     -    CALL HALT('TRSPEC - Frequency values too closely spaced')
 320  CONTINUE


      RETURN

      END
