*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE TPLSLV (R, G, N, C, IER)
*
*
* Purpose:
*     Solve a Toeplitz set of equations
*
*
* Description:
*     This subroutine solves the set of equations
*
*       R c = g
*
*     where R is an N by N symmetric Toeplitz matrix and c and r are N
*     element column vectors.  The Toeplitz matrix R has elements which
*     are equal along the diagonals, i.e. R(i,j)=f(abs(i-j)).
*
*     This subroutine uses Levinson's method to calculate the solution.
*     In the case that either numerical instability or an inappropriate
*     set of matrix coefficients results in a matrix which is not
*     positive definite or singular, an error code is returned.
*
*     This routine uses:
*           2N-1    divides, and
*         2N**2-3N  multiplies and adds.
*
*     Reference:
*       E.A. Robinson, "Multichannel Time Series Analysis with
*       Digital Computer Programs", Holden-Day, 1967, p. 44.
*
*
* Parameters:
* D ->  R      - Vector specifying the first column of the Toeplitz
*                matrix (double precision)
* D ->  G      - Righthand side vector (double precision)
* I ->  N      - Number of equations (maximum 50)
* D <-  C      - Solution vector (double precision)
* I <-  IER    - Error flag, 0 - no error
*                            1 - matrix not positive definite
*
*
* Routines required:
*     None
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:34:45 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE TPLSLV (R, G, N, C, IER)


      INTEGER NMAX
      PARAMETER (NMAX=50)


      INTEGER N,IER
      INTEGER K,L,I

      DOUBLE PRECISION R(N),G(N),C(N)
      DOUBLE PRECISION PRERR,SUM,RF,CK
      DOUBLE PRECISION PRDCOF(NMAX-1)


      IER=0
      PRERR=R(1)
      IF (PRERR.LE.0D0) THEN
        IER=1
        GO TO 900
      END IF

      C(1)=G(1)/PRERR

      DO 180 K=1,N-1

        L=K
        SUM=-R(K+1)
        DO 120 I=1,K-1
          SUM=SUM+R(L)*PRDCOF(I)
          L=L-1
 120    CONTINUE

        RF=SUM/PRERR

* Calculate the prediction error (equivalent to
* PRERR = PRERR * (1-RF**2))
* A change in sign of PRERR means that RF has a magnitude
* greater than unity (corresponding to a non-positive definite system)
        PRERR=PRERR-RF*SUM
        IF (PRERR.LE.0D0) THEN
          IER=1
          GO TO 900
        END IF

        PRDCOF(K)=-RF

        L=K-1
        DO 140 I=1,K/2
          SUM=PRDCOF(I)+RF*PRDCOF(L)
          IF (I.NE.L) PRDCOF(L)=PRDCOF(L)+RF*PRDCOF(I)
          PRDCOF(I)=SUM
          L=L-1
 140    CONTINUE

        L=K+1
        SUM=G(K+1)
        DO 160 I=1,K
          SUM=SUM-C(I)*R(L)
          L=L-1
 160    CONTINUE

        CK=SUM/PRERR
        C(K+1)=CK

        L=K
        DO 170 I=1,K
          C(I)=C(I)-CK*PRDCOF(L)
          L=L-1
 170    CONTINUE

 180  CONTINUE


 900  RETURN

      END
