*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     DOUBLE PRECISION FUNCTION TACORR (T, F, FRESP, NVAL)
*
*
* Purpose:
*     Evaluate a correlation, given spectrum values
*
*
* Description:
*     This function returns the value of an autocorrelation function
*     at a given time lag.  The correlation function is calculated as
*     the Fourier transform of a power spectrum.  The corresponding
*     correlation is real and symmetrical about time lag zero.
*
*     The power spectrum is specified in tabular form.  A piecewise
*     monotonic cubic interpolation is used between the given values.
*     The power spectrum is specified for non-negative frequencies.
*     The cubic interpolant has zero slope at the first and last points
*     specified.  Additional sinusoidal components can be specified for
*     the power spectrum.
*
*
* Parameters:
* D <-  TACORR - Returned function value (double precision)
*
* D ->  T      - Time lag value at which the correlation is to be
*                evaluated (double precision)
* R ->  F      - Array of normalized frequencies.  The frequency values
*                must be non-negative. The normalizing frequency is the
*                frequency of the input sequence before interpolation.
*                The first NVAL(1) values must be in increasing order.
*                These are the frequencies of samples of the continuous
*                power spectrum.  The next NVAL(2) values are the
*                frequencies of the sinusoidal components.
* R ->  PSD    - Array of power spectral density values.  The value
*                PSD(i) is the power spectral density at frequency
*                F(i).  Each PSD(i) must be non-negative.  The first
*                NVAL(1) values are samples of the continuous power
*                spectrum.  The next NVAL(2) values are the powers of
*                the sinusoidal components.
* I ->  NVAL   - Number of frequency and power spectrum density values.
*                This is a two element array.  NVAL(1) gives the number
*                of samples of the continuous power spectrum (at least
*                2). NVAL(2) gives the number of sinusoidal components.
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     TRESPF - Evaluate a time response, given freq. response values
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:33:38 $
*
*
*-----------------------------------------------------------------------

      DOUBLE PRECISION FUNCTION TACORR (T, F, PSD, NVAL)


      DOUBLE PRECISION PI2
      PARAMETER (PI2=6.283 185 307 179 586 476 9D0)

      INTEGER NVAL(2)
      INTEGER I

      REAL F(*),PSD(*)

      DOUBLE PRECISION T
      DOUBLE PRECISION CORR
      DOUBLE PRECISION TRESPF


* Evaluate the correlation function
      IF (NVAL(1).GT.0) THEN
        CORR=TRESPF(T,F,PSD,NVAL(1))
      ELSE
        CORR=0.0
      END IF
      DO 100 I=NVAL(1)+1,NVAL(1)+NVAL(2)
        IF (F(I).LT.0.0)
     -    CALL HALT('TACORR - Error, negative frequency')
        IF (PSD(I).LT.0.0)
     -    CALL HALT('TACORR - Error, negative sinusoid amplitude')
        CORR=CORR + PSD(I)*COS(PI2*F(I)*T)
 100  CONTINUE

* Return the function value
      TACORR=CORR


      RETURN

      END
