*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION SNRINT (H, NTAP, IR, F, PSD, NVAL)
*
*
* Purpose:
*     Calculate the mean-square interpolation error
*
* Description:
*     This function calculates the mean-square interpolation error for
*     an interpolating filter or a Nyquist filter given the power
*     spectrum of the input signal.  The power spectrum of the input
*     signal is specified in terms of a continuous component and
*     sinusoidal components. The continuous component is given as a
*     tabulated function. A piecewise monotonic cubic interpolation is
*     used between the given values. The sinusoidal components are
*     specified by frequency and power. The continuous power spectrum
*     is given as pairs of values: a normalized frequency and the
*     corresponding power spectral density value.  The frequencies
*     must be non-negative and in increasing order.  The continuous
*     power spectrum is assumed to be zero outside the given range of
*     frequencies.  The cubic interpolant has zero slope at the first
*     and last points of the continuous portion of the power spectrum.
*     Sinusoidal components of the power spectrum are specified as a
*     normalized frequency (non-negative) and the corresponding power.
*
*
* Parameters:
* R <-  SNRINT - Resulting normalized mean-square interpolation error.
*                The interpolation error depends on the relative phase
*                of the sample being interpolated.  SNRINT is the
*                average of the normalized mean-square errors for each
*                possible sample phase.  Normalization is with respect
*                to the power of the input signal as calculated from
*                the power spectral density.
*
* R ->  H      - Filter coefficients.  The middle of the filter is the
*                reference time point.
* I ->  NTAP   - Number of filter coefficients (maximum 50*IR+1).
* I ->  IR     - Interpolation ratio
* R ->  F      - Array of normalized frequencies.  The frequency values
*                must be non-negative. The normalizing frequency is the
*                frequency of the input sequence before interpolation.
*                The first NVAL(1) values must be in increasing order.
*                These are the frequencies of samples of the continuous
*                power spectral density.  The next NVAL(2) values are
*                the frequencies of the sinusoidal components.
* R ->  PSD    - Array of power spectrum values. The value PSD(i) gives
*                the power spectrum at frequency F(i). Each PSD(i) must
*                be non-negative.  The first NVAL(1) values are samples
*                of the continuous power spectral density.  The next
*                NVAL(2) values are the powers of the sinusoidal
*                components.
* I ->  NVAL   - Number of frequency and power spectrum values.  This
*                is a two element array.  NVAL(1) gives the number of
*                samples of the continuous power spectrum (at least 2).
*                NVAL(2) gives the number of sinusoidal components.
*
*
* Routines required:
*     FLERR2 - Calculate the mean-square filtering error
*     HALT   - Print an error message, stop with error status set
*     ICEIL  - Calculate the ceiling function of a ratio of integers
*     IFLOOR - Calculate the floor function of a ratio of integers
*     TACORR - Evaluate a correlation, given spectrum values
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:32:27 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION SNRINT (H, NTAP, IR, F, PSD, NVAL)


      INTEGER NMAX
      PARAMETER (NMAX=50)

      INTEGER NTAP,IR,NVAL(2)
      INTEGER IS,NR,NT,NH,IVL,IVR,IUL,IUR
      INTEGER NM,IST,I,LL,LU,J,NN,M,K
      INTEGER IC
      INTEGER MODP,N1,N2
      INTEGER IFLOOR,ICEIL

      REAL H(0:*),F(*),PSD(*)

      DOUBLE PRECISION HS(0:NMAX-1),P(0:NMAX-1),RXX(0:NMAX-1)
      DOUBLE PRECISION SUME
      DOUBLE PRECISION TACORR,FLERR2
      DOUBLE PRECISION R


* Statement function to calculate the correlation function (integer
* argument)
      R(IC) = TACORR(DBLE(IC)/NR,F,PSD,NVAL)

* Statement function to calculate the modulo function with a positive
* remainder (N1 of any sign, N2 positive)
      MODP(N1,N2)=N1-N2*IFLOOR(N1,N2)


* Let the number of taps be odd, Nt=2*Nh+1.
* The interpolating filter is divided into NR subfilters,
*   hs(m,i) = h(m*NR+i) , where 0 <= i < NR

* The case of an even number of coefficients is handled by conceptually
* creating an interpolating filter with an interpolating ratio of 2*IR.
* This double length filter (adding one coefficient between every one
* of the original filter) has an odd length.  Only coefficients for
* every second subfilter are used.

* NR  - Effective interpolating ratio
* IS  - Subsampling factor, 1 for odd length filters
*                           2 for even length filters
* NT  - Number of taps in the filter with interpolating ratio NR,
*       (IS-1) extra coefficients between each real coefficient

      IF (MOD(NTAP,2).EQ.0) THEN
        IS=2
      ELSE
        IS=1
      END IF
      NR=IS*IR
      NT=NTAP+(IS-1)*(NTAP-1)

* Calculate the starting and ending coefficients
* The filter has coefficient indices -NH,...,-1,0,+1,...,+NH
*   -NH = IUL*NR + IVL, where 0 <= IVL < NR
*   +NH = IUR*NR + IVR, where 0 <= IVR < NR
      NH=(NT-1)/2
      IVL=MODP(-NH,NR)
      IVR=MODP(NH,NR)
      IUL=IFLOOR(-NH-IVL,NR)
      IUR=IFLOOR(NH-IVR,NR)

* Find the length of the longest subfilter
      NM=ICEIL(NTAP,IR)
      IF (NM.GT.NMAX)
     -  CALL HALT('SNRINT - Too many coefficients')

* Fill in the autocorrelation vector
      DO 120 M=0,NM-1
        RXX(M)=R(M*NR)
 120  CONTINUE

* Loop over the subfilter indices
      IF (IS.EQ.1) THEN
        IST=0
      ELSE
        IST=1
      END IF

      SUME=0D0
      DO 400 I=IST,NR-1,IS

* Calculate the lengths of the I'th subfilter
        LL=IUL
        IF (I.LT.IVL) LL=LL+1
        LU=IUR
        IF (I.GT.IVR) LU=LU-1

* Fill in the autocorrelation vector
* Fill in the right-hand side vector, p[m]
* Interpolating filter, interpolating ratio NR
*  index j: -NH, ... , -1, 0, +1, ... , +NH
* Interpolating filter, interpolating ratio IR
*  index k: 0, 1, ... , NTAP-1
* The mapping between these is k=(j+NH)/IS
        J=LL*NR+I
        K=(J+NH)/IS
        NN=LU-LL+1
        DO 320 M=0,NN-1
          P(M)=R(J)
          HS(M)=H(K)
          J=J+NR
          K=K+IR
 320    CONTINUE

* The resulting mean-square error for the i'th sub-filter can be
* expressed as
*   E = r(0) - 2 hs'[i] p[i] + hs'[i] R hs[i] ,
* where the matrix R has elements R(m,k) = r((m-k)*NR). This expression
* is general in that the vector hs[i] need not be the "optimal" value.
        SUME=SUME+FLERR2(RXX,HS,P,NN)

 400  CONTINUE

* Return the average mean-square error
      SNRINT=SUME/(IR*RXX(0))


      RETURN

      END
