*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REMEZ (NGRID, XGRID, DES, WT, DLIML, DLIMU,
*                       X, Y, LIM, NEXT, NCOF, DEVS, IER)
*
*
* Purpose:
*     Remez exchange algorithm for filter design
*
*
* Description:
*     This routine finds a minimum maximum deviation approximation to a
*     function, subject to constraints of the allowed deviation.  The
*     function is specified by the desired function values and weights
*     on a dense grid.  The algorithm minimizes the Chebyshev error by
*     determining the best locations of the extremal points (points of
*     maximum error). The returned values are the extremal points of
*     the approximation.
*
*
* Parameters:
* I ->  NGRID  - Number of elements in each of XGRID, DES, WT, DLIML,
*                and DLIMU
* R ->  XGRID  - Array of abscissa values (grid points).  These should
*                be in decreasing order in the range [-1,+1].
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid, DLIML <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid, DLIMU >= DES(i)
* D <-  X      - Double precision array of abscissa values at the
*                extremal points.  Each X(i) takes on a unique value
*                from XGRID(k).
* D <-  Y      - Double precision array of ordinate values at the
*                extremal points.  For the general case, NEXT = NCOF+1,
*                the NEXT extrema are constrained in such a way as to
*                allow an NCOF term polynomial to pass through the NEXT
*                points.  For the special case, NEXT = NGRID < NCOF+1,
*                an NEXT term polynomial will pass exactly through the
*                NGRID desired response values giving DEVS = 0.
* I <-  LIM    - Type of extremum,
*                -2 - Lower constraint limit
*                -1 - Lower ripple
*                +1 - Upper ripple
*                +2 - Upper constraint limit
* I <-  NEXT   - Number of extrema, NEXT=MIN(NCOF+1,NGRID)
* I ->  NCOF   - Number of polynomial coefficients to be used in
*                the approximation (maximum 128)
* R <-  DEVS   - Resultant deviation (relative to unity weighting)
* I <-  IER    - Error parameter coded as follows.  This routine will
*                return extremal values even for the nonzero error
*                codes.
*                0 - No error
*                1 - Remez algorithm failed to converge, the deviation
*                    is not monotonically increasing
*                2 - Remez algorithm failed to converge, too many
*                    iterations
*                3 - Constraints too tight for the given filter order
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     LGCOEF - Generate Lagrange interpolation coefficients
*     REMDEV - Calculate the deviation for a minimax approximation
*     REMEXC - Extremal search for filter design
*     WARN   - Print a warning message on the standard error unit
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:32:28 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REMEZ (NGRID, XGRID, DES, WT, DLIML, DLIMU,
     -                  X, Y, LIM, NEXT, NCOF, DEVS, IER)


      INTEGER LLIM,LDEV,UDEV,ULIM,MAXITR,NCFMAX,MAXEXT
      PARAMETER (LLIM=-2,LDEV=-1,UDEV=+1,ULIM=+2)
      PARAMETER (MAXITR=40,NCFMAX=128,MAXEXT=NCFMAX+1)

      INTEGER NGRID,LIM(*),NEXT,NCOF,IER
      INTEGER I,ITER,L,NCHNGE
      INTEGER IEXT(MAXEXT)

      REAL XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID),DEVS
      REAL DEVL,DENS

      DOUBLE PRECISION X(*),Y(*)
      DOUBLE PRECISION AD(MAXEXT)


      DEVL=-1.

* Initial guess for the extremal frequencies (equally spaced
* along the frequency axis)
      NEXT=MIN(NGRID,NCOF+1)
      DENS=REAL(NGRID-1)/MAX(NEXT-1,1)
      DO 100 I=1,NEXT
        IEXT(I)=NINT((I-1)*DENS)+1
 100  CONTINUE

* Set up the alternating limit array
* The initial setting might be inverted, but is corrected in the
* first iteration
      LIM(1)=LDEV
      DO 120 I=2,NEXT
        IF (LIM(I-1).EQ.LDEV) THEN
          LIM(I)=UDEV
        ELSE
          LIM(I)=LDEV
        END IF
 120  CONTINUE


*********************
      DO 400 ITER=1,MAXITR

* Store the abscissa values for the Lagrange interpolation
        DO 220 I=1,NEXT
          X(I)=XGRID(IEXT(I))
 220    CONTINUE

* Calculate the Lagrange interpolation coefficients to
* interpolate NEXT values
        CALL LGCOEF(AD,X,NEXT)

* Calculate the deviation:
* Given a set of NEXT distinct points, an NEXT-1 coefficient polynomial
* can give an error curve which exhibits equal amplitude alternations
* (in a weighted sense) about the desired value.
* In such a formulation, the free parameters are the NEXT-1
* coefficients and the deviation giving a total of NEXT linear
* equations.
        CALL REMDEV(AD,NEXT,NGRID,DES,WT,DLIML,DLIMU,
     -              IEXT,LIM,NCOF,DEVS,IER)
        IF (IER.NE.0) GO TO 500

* Correct the sequence of deviations if necessary (first iteration
* only)
        IF (DEVS.LT.0.0) THEN
          IF (ITER.NE.1)
     -      CALL HALT('REMEZ - Change in sign of deviation')
          DO 240 I=1,NEXT
            IF (LIM(I).EQ.LDEV) THEN
              LIM(I)=UDEV
            ELSE IF (LIM(I).EQ.UDEV) THEN
              LIM(I)=LDEV
            END IF
 240      CONTINUE
          DEVS=-DEVS
        END IF

* Calculate the ordinates for the Lagrange interpolation.
* The polynomial that passes through the NEXT alternations with
* deviation as calculated above has only NEXT-1 terms.
* However, if we keep the ordinate values for the NEXT term
* interpolation at the NEXT alternation values, the Lagrange
* coefficients for the NEXT term case can be used to interpolate
* the NEXT-1 term polynomial.
        DO 280 I=1,NEXT
          L=IEXT(I)
          IF (LIM(I).EQ.LDEV) THEN
            Y(I)=DES(L)-DEVS/WT(L)
          ELSE IF (LIM(I).EQ.UDEV) THEN
            Y(I)=DES(L)+DEVS/WT(L)
          ELSE IF (LIM(I).EQ.LLIM) THEN
            Y(I)=DLIML(L)
          ELSE
            Y(I)=DLIMU(L)
          END IF
 280    CONTINUE

* Check the convergence
        IF (DEVS.LT.DEVL) THEN
          IER=1
          CALL WARN('REMEZ - Fails to converge (Deviation non-'//
     -              'increasing)')
          GO TO 500
        END IF
        DEVL=DEVS

* Remez exchange algorithm
        CALL REMEXC(AD,X,Y,NEXT,NGRID,XGRID,DES,WT,DLIML,DLIMU,
     -              DEVS,IEXT,LIM,NCHNGE)
        IF (NCHNGE.LE.0) GO TO 500

 400  CONTINUE
*********************

* Failed to converge
      IER=2
      CALL WARN('REMEZ - Fails to converge (too many iterations)')


 500  RETURN

      END
