*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE REDMPH (NAMEFL, NTAP, NN, NF, FREQ, WEIGHT, GRIDDD)
*
*
* Purpose:
*     Read Nyquist filter specifications (factored filter)
*
*
* Description:
*     This subroutine issues prompts and then reads filter design
*     specifications.  A maximum of 200 frequency points may be
*     specified.  If an error is detected, a message is signalled and
*     execution is halted.
*
*
* Parameters:
* C <-  NAMEFL - Character string specifying the coefficient file name.
*                NAMEFL may be all blank.
* I <-  NTAP   - Number of filter coefficients (maximum 128)
* I <-  NN     - Zero crossing interval
* I <-  NF     - Number of frequency points
* R <-  FREQ   - Array of normalized frequencies.  These are in
*                increasing order.
* R <-  WEIGHT - Array of desired weightings
* R <-  GRIDD  - Grid density.  GRIDD is zero if unspecified, or equal
*                to -1, if MAXIMUM is specified.
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     KEYUPC - Match keyword strings (case insensitive)
*     LENNUL - Find the length of a null terminated string
*     RDCSTR - Separate comma/whitespace delimited substrings
*     RDIVA1 - Decode an integer value
*     RDLINE - Read a line of input from standard input
*     RDRVA1 - Decode a real value
*     RDRVA2 - Decode two real values
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:01:41 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE REDMPH (NAMEFL, NTAP, NN, NF, FREQ, WEIGHT, GRIDD)


      INTEGER NFMAX,MXPT
      PARAMETER (NFMAX=128,MXPT=200)

      CHARACTER*(*) NAMEFL
      CHARACTER*80 LINE,SLINE(3)
      CHARACTER*8 MAXTAB(1)

      INTEGER NTAP,NN,NF,I,K
      INTEGER NCHR,ISRC,IER,N,NCHRN
      INTEGER LENNUL,KEYUPC

      REAL SFREQ,GRIDD,FP
      REAL FREQ(*),WEIGHT(*)

      DATA MAXTAB/'MAX*IMUM'/


* Read the zero crossing interval
      CALL RDLINE('Zero crossing interval: ',LINE,NCHR,ISRC)
        IF (NCHR.EQ.0) CALL HALT('REDMPH - Insufficient data')
      CALL RDIVA1(LINE,NN,IER)
        IF (IER.NE.0) CALL HALT('REDMPH - Data error')
        IF (NN.LE.0)
     -    CALL HALT('REDMPH - Invalid zero crossing interval')

* Read the number of coefficients
      CALL RDLINE('No. Coeff, Sampling rate: ',LINE,NCHR,ISRC)
        IF (NCHR.EQ.0) CALL HALT('REDMPH - Insufficient data')
      CALL RDCSTR(LINE,1,3,SLINE,N)
        IF (N.LE.0)
     -    CALL HALT('REDMPH - Invalid number of parameters')
      NCHR=LENNUL(SLINE(1))
      CALL RDIVA1(SLINE(1)(1:NCHR),NTAP,IER)
        IF (IER.NE.0) CALL HALT('REDMPH - Data error')
        IF (NTAP.LE.0)
     -    CALL HALT('REDMPH - Invalid number of coefficients')
        IF (NTAP.GT.NFMAX)
     -    CALL HALT('REDMPH - Too many coefficients')

* Decode the optional parameters SFREQ and GRIDD
      SFREQ=1.0
      IF (N.GE.2) THEN
        NCHR=LENNUL(SLINE(2))
        IF (NCHR.GT.0) THEN
          CALL RDRVA1(SLINE(2)(1:NCHR),SFREQ,IER)
          IF (IER.NE.0) CALL HALT('REDMPH - Data error')
          IF (SFREQ.LE.0)
     -      CALL HALT('REDMPH - Invalid sampling frequency')
        END IF
      END IF

      GRIDD=0.0
      IF (N.EQ.3) THEN
        NCHR=LENNUL(SLINE(3))
        IF (NCHR.GT.0) THEN
          N=KEYUPC(SLINE(3)(1:NCHR),MAXTAB,1)
          IF (N.EQ.1) THEN
            GRIDD=-1.
          ELSE
            CALL RDRVA1(SLINE(3)(1:NCHR),GRIDD,IER)
            IF (IER.NE.0) CALL HALT('REDMPH - Data error')
            IF (GRIDD.LE.0.0)
     -        CALL HALT('REDMPH - Invalid grid density')
          END IF
        END IF
      END IF

* Read the coefficient file name
      CALL RDLINE('Coefficient file: ',NAMEFL,NCHRN,ISRC)
        IF (ISRC.LT.0) CALL HALT('REDMPH - Insufficient data')

      FP=-1.
      DO 140 I=1,MXPT+1
        CALL RDLINE('  Stopband (Freq., Weight): ',LINE,NCHR,ISRC)
          IF (NCHR.LE.0) GO TO 160

* Frequency point
        IF (I.GT.MXPT)
     -    CALL HALT('REDMPH - Too many specification records')
        K=I
        CALL RDRVA2(LINE,FREQ(K),WEIGHT(K),IER)
        IF (IER.NE.0) CALL HALT('REDMPH - Data error')
        FREQ(K)=FREQ(K)/SFREQ

* Error checks
        IF (FREQ(K).LT.0.0 .OR. FREQ(K).GT.0.5)
     -    CALL HALT('REDMPH - Invalid frequency')
        IF (FREQ(K).LE.FP)
     -    CALL HALT('REDMPH - Frequency values not in '//
     -              'increasing order')
        IF (WEIGHT(K).LE.0.0)
     -    CALL HALT('REDMPH - Weight value not positive')

        IF (FREQ(K).GE.0.5) GO TO 160
        FP=FREQ(K)
 140  CONTINUE

 160  NF=K
      IF (NF.LE.0)
     -   CALL HALT('REDMPH - No frequency bands specified')


      RETURN

      END
