*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE PRTFIR (ITYPE, DEVS, IBAND, NBANDS,
*                        FREQ, VAL, WEIGHT, VLIML, VLIMU)
*
*
* Purpose:
*     Print FIR filter design parameters
*
*
* Description:
*     This subroutine prints the specifications and approximation error
*     for each specified frequency point for an FIR filter.
*
*
* Parameters:
* I ->  ITYPE  - Filter type coded as follows.
*                1 - Multiple passband/stopband filter
*                2 - Multiple passband/stopband filter
*                    (sin(X)/X compensation)
*                3 - Differentiator
*                4 - Hilbert transform filter
* R ->  DEVS   - Deviation corresponding to unit weight
* I ->  IBAND  - Frequency band specification index.  The array FREQ
*                specifies the frequency bands.
*                 Band    Lower Band Edge          Upper Band Edge
*                  1     FREQ(1)                  FREQ(IBAND(1))
*                  2     FREQ(IBAND(1)+1)         FREQ(IBAND(2))
*                 ...        ...                    ...
*                NBANDS  FREQ(IBAND(NBANDS-1)+1)  FREQ(IBAND(NBANDS))
*                VAL(i) and WEIGHT(i) specify the desired value and
*                weight at frequency FREQ(i).
* I ->  NBANDS - Number of bands specified
* R ->  FREQ   - Array of normalized frequencies.  These are in
*                increasing order.
* R ->  VAL    - Array of desired amplitude values (desired slopes for
*                differentiators)
* R ->  WEIGHT - Array of desired weightings.  For differentiators,
*                the given weighting applies to the slope.
* R ->  VLIML  - Array of lower constraints. If the constraint takes on
*                the value -1E20, it is assumed not to have been
*                specified.
* R ->  VLIMU  - Array of upper constraints. If the constraint takes on
*                the value +1E20, it is assumed not to have been
*                specified.
*
*
* Routines required:
*     None
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:26:41 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE PRTFIR (ITYPE, DEVS, IBAND, NBANDS,
     -                   FREQ, VAL, WEIGHT, VLIML, VLIMU)


      INTEGER NKEY,BPF,REC,DIF,HIL
      PARAMETER (NKEY=4,BPF=1,REC=2,DIF=3,HIL=4)

      CHARACTER*9 CSTR

      LOGICAL LIMS

      INTEGER ITYPE,IBAND(*),NBANDS
      INTEGER IU,K,IL,I

      REAL DEVS,FREQ(*),VAL(*),WEIGHT(*),VLIML(*),VLIMU(*)
      REAL DBV,X,DEV,DEVDB

      CHARACTER FLTTYP(NKEY)*40
      DATA FLTTYP/'Bandpass Filter:',
     -            'Bandpass Filter (sin(x)/x compensation):',
     -            'Differentiator:',
     -            'Hilbert Transform Filter:'/


* Statement function to convert to decibels
      DBV(X)=20.0*LOG10(X)


* Title
      WRITE (UNIT=*,FMT=2000) FLTTYP(ITYPE)
      IF (ITYPE.EQ.DIF) THEN
        WRITE (UNIT=*,FMT=2100)
      ELSE IF (ITYPE.EQ.HIL) THEN
        WRITE (UNIT=*,FMT=2200)
      ELSE
        WRITE (UNIT=*,FMT=2300)
      END IF

* Results by band
      IU=0
      DO 120 K=1,NBANDS
        IL=IU+1
        IU=IBAND(K)
        WRITE (UNIT=CSTR,FMT=4000) K
        DO 100 I=IL,IU

* Calculate the weighted deviation
          DEV=DEVS/WEIGHT(I)
          LIMS=VLIML(I).NE.-1E20 .OR. VLIMU(I).NE.+1E20
          IF (LIMS)
     -      DEV=MAX(MIN(DEV,VAL(I)-VLIML(I)),
     -              MIN(DEV,VLIMU(I)-VAL(I)))
          IF (DEV.GT.0.0) THEN
            IF (VAL(I).NE.0.0) THEN
              DEVDB=DBV((DEV+VAL(I))/VAL(I))
            ELSE
              DEVDB=DBV(DEV)
            END IF
          ELSE
            DEVDB=-1E20
          END IF

* Band specifications
          IF (ITYPE.NE.DIF .AND. ITYPE.NE.HIL) THEN
            IF (LIMS) THEN
              WRITE (UNIT=*,FMT=2400) CSTR,FREQ(I),VAL(I),
     -                                WEIGHT(I),VLIML(I),VLIMU(I),
     -                                DEV,DEVDB
            ELSE
              WRITE (UNIT=*,FMT=2500) CSTR,FREQ(I),VAL(I),
     -                                WEIGHT(I),DEV,DEVDB
            END IF
          ELSE
            IF (LIMS) THEN
              WRITE (UNIT=*,FMT=2400) CSTR,FREQ(I),VAL(I),
     -                                WEIGHT(I),VLIML(I),VLIMU(I),
     -                                DEV
            ELSE
              WRITE (UNIT=*,FMT=2500) CSTR,FREQ(I),VAL(I),
     -                                WEIGHT(I),DEV
            END IF
          END IF
          CSTR=' '

 100    CONTINUE
 120  CONTINUE


      RETURN

 2000 FORMAT (/1X,A)
 2100 FORMAT ('           Freq.   Slope   Weight ',
     -                 ' Lower Lim. Upper Lim.  Deviation')
 2200 FORMAT ('           Freq.   Value   Weight ',
     -                 ' Lower Lim. Upper Lim.  Deviation')
 2300 FORMAT ('           Freq.   Value   Weight ',
     -                 ' Lower Lim. Upper Lim.  Deviation  Dev. dB')
 2400 FORMAT (A,F7.4,1PG11.3,1PG9.2,    1PG11.3,      1PG11.3,
     -          1PG9.2, 0PF8.2)
 2500 FORMAT (A,F7.4,1PG11.3,1PG9.2,'   ----    ','   ----    ',
     -          1PG9.2, 0PF8.2)
 4000 FORMAT (' Band',I3,':')

      END
