*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE NYQSLV (G, NG, NN, F, NF)
*
*
* Purpose:
*     Solve for a cascade Nyquist filter
*
*
* Description:
*     This routine solves for the coefficients of the linear phase FIR
*     filter F(z) which will force the response of the cascade filter
*     H(z) = F(z) G(z) to be a Nyquist response.  A Nyquist response
*     h(n) has regular zero crossings every N samples,
*       h(0)   = 1/N,
*       h(n*N) = 0,   for n ...,-3,-2,-1,+1,+2,+3,...
*
*
* Parameters:
* D ->  G      - Array of double precision filter coefficients (NG
*                values).  The corresponding filter is linear phase,
*                i.e., the coefficients satisfy G(i)=G(-i), for
*                i=-(NG-1), ... , NG-1.  The coefficients G(0), ... ,
*                G(NG-1) are given as inputs.
* I ->  NG     - Number of input filter coeffients
* I ->  NN     - Zero crossing interval
* D <-  F      - Resulting array of double precision filter
*                coefficients (NF values).  The corresponding filter is
*                linear phase, i.e. the values obey F(i)=F(-i), for
*                i=-(NF-1), ... , NF-1.  The coefficients returned are
*                F(0), ... , F(NF-1).
* I ->  NF     - Number of output filter coefficients.  The values NG
*                and NF must be compatible.  Specifically, the overall
*                cascade filter has length NH=2*(NF+NG)-3.  Let
*                L0=(NH-1)/(2*NN).  Then NF must equal L0+1.
*
*
* Routines required:
*     DLNEQN - Solve a general set of linear equations
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:26:42 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE NYQSLV (G, NG, NN, F, NF)


      INTEGER NHMMAX,NHMAX,L0MAX
      PARAMETER (NHMMAX=128,NHMAX=2*NHMMAX-1,L0MAX=(NHMAX-1)/2)

      INTEGER NG,NN,NF
      INTEGER L0,L1,NH,I,L1S,L0S,K,J

      DOUBLE PRECISION G(0:NG-1),F(0:NF-1)
      DOUBLE PRECISION A(0:L0MAX,0:L0MAX),C(0:L0MAX),T


* Initialization
      L0=NF-1
      L1=NG-1
      NH=2*(L0+L1)+1
      IF (L0.NE.(NH-1)/(2*NN) .OR. L1.NE.(NH-1)/2-L0)
     -  CALL HALT('NYQSLV - Incompatible filter lengths')

* If G has trailing zeros, the equations to be solved can be
* overdetermined.  This condition is handled by solving for the
* minimum length filter satisfying the Nyquist criterion and
* then appending zeros.
      DO 120 I=L1,0,-1
        IF (G(I).NE.0.0) GO TO 140
 120  CONTINUE
      CALL HALT('NYQSLV - All input coefficients are zero')
 140  L1S=I
      L0S=(L1S-1)/MAX(NN-1,1)

* Fill in the coefficient matrix of the linear system of equations
      DO 220 I=0,L0S
        K=I*NN
        IF (K.LE.L1S) THEN
          A(I,0)=G(K)
        ELSE
          A(I,0)=0D0
        END IF
        DO 200 J=1,L0S
          T=0D0
          IF (ABS(K-J).LE.L1S) T=G(ABS(K-J))
          IF (K+J.LE.L1S) T=T+G(K+J)
          A(I,J)=T
 200    CONTINUE
 220  CONTINUE

      C(0)=1D0/NN
      DO 240 I=1,L0S
        C(I)=0D0
 240  CONTINUE

* Solve the equations for the coefficients of F(z)
      CALL DLNEQN(A,L0MAX+1,L0S+1,C,F)

* Append zeros if necessary
      DO 320 I=L0S+1,L0
        F(I)=0.0
 320  CONTINUE


      RETURN

      END
