*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE NYQFLT (NGRID, FGRID, XGRID, DES, WT, DLIML, DLIMU,
*                        H, NH, NN, DEVS, IER)
*
*
* Purpose:
*     Design a Nyquist filter
*
*
* Description:
*     This routine designs a linear phase FIR Nyquist filter.
*
*
* Parameters:
* I ->  NGRID  - Number of elements in each of FGRID, XGRID, DES, WT,
*                DLIML, and DLIMU (maximum 1000)
* R ->  FGRID  - Array of normalized frequency values.  These values
*                values must be in increasing order in the range
*                [0,1/2].
* R ->  XGRID  - Array of abscissa values (grid points).  These should
*                be in decreasing order in the range [-1,+1].  The
*                element XGRID(i) is assumed to be equal to
*                COS(2*Pi*FGRID(i)).
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid,
*                DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid,
*                DLIMU(i) >= DES(i)
* R <-  H      - Array of resultant filter coefficients (double
*                precision) for the Nyquist filter
* I ->  NH     - Number of filter coefficients desired (odd, maximum
*                255)
* I ->  NN     - Zero crossing interval
* R <-  DEVS   - Resultant stopband deviation from the desired
*                specifications.  This value is unweighted.  The
*                deviation at a given frequency is obtained by dividing
*                by the weight at that frequency.
* I <-  IER    - Error parameter coded as follows.
*                0  - No error
*                3  - Too many iterations in the Nyquist design
*                For these cases, this routine returns coefficient
*                values.  For all other errors, an error message is
*                printed and execution is halted.
*
*
* Routines required:
*     AMSFIR - Find the amplitude response of a symmetric FIR filter
*     DCONLP - Convolve symmetric responses
*     IMPULS - Find the impulse response for an FIR filter
*     NYQSLV - Solve for a cascade Nyquist filter
*     REMCOF - Generate the sum of cosines coefficients for a filter
*     REMEZ  - Remez exchange algorithm for filter design
*     WARN   - Print a warning message on the standard error unit
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:01:01 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE NYQFLT (NGRID, FGRID, XGRID, DES, WT, DLIML, DLIMU,
     -                   H, NH, NN, DEVS, IER)


      INTEGER NGMAX,NHMAX,L0MAX,L1MAX,NH1MAX,NCFMAX,MAXEXT,NIMAX
      PARAMETER (NGMAX=1000,NHMAX=255,
     -           L0MAX=(NHMAX-1)/2,L1MAX=(NHMAX-1)/2,
     -           NH1MAX=2*L1MAX+1,
     -           NCFMAX=(NH1MAX+1)/2,MAXEXT=NCFMAX+1,NIMAX=10)

      INTEGER NGRID,NH,NN,IER
      INTEGER L0,L1,NH0,NH1,ICASE,NCOF,I,NEXT,ITER,NCOFS,NC
      INTEGER LIM(MAXEXT)

      LOGICAL CONVRG

      REAL FGRID(NGRID),XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID),DEVS
      REAL WTH(NGMAX)

      DOUBLE PRECISION H(NH)
      DOUBLE PRECISION ALPHA(NCFMAX),X(MAXEXT),XS(MAXEXT),
     -     Y(MAXEXT),H0(2*L0MAX+1),H1(NH1MAX)
      DOUBLE PRECISION AMSFIR


* Array sizes;
*   H(z) = H0(z) H1(z),
*
*   L0=(NH-1)/(2*NN), L1=(NH-1)/2 - L0
*   Largest L0 occurs for NN=1, L0max=(NHmax-1)/2
*   Largest L1 occurs for NN large, L1max=(NHmax-1)/2.
*
*   In terms of L0 and L1,
*   NH1=2*L1+1, NH0=2*L0+1
*
*   Largest array dimensions,
*   NH1max=2*L0max+1, NH0max=2*L0max+1,

* Filter lengths
      L0=(NH-1)/(2*NN)
      L1=(NH-1)/2-L0
      NH0=2*L0+1
      NH1=2*L1+1
      IF (MOD(NH1,2).NE.0) THEN
        ICASE=1
        NCOF=(NH1-1)/2+1
      ELSE
        ICASE=2
        NCOF=NH1/2
      END IF

* Initialization
      DO 200 I=1,NH0
        H0(I)=0D0
 200  CONTINUE
      H0(L0+1)=1D0

      NEXT=MIN(NCOF+1,NGRID)
      DO 220 I=1,NEXT
        XS(I)=0D0
 220  CONTINUE

* The Nyquist filter is expressed as H0(z) H1(z).  H1(z) is a filter
* with all of its zeros in the stopband.  H0(z) has the passband
* zeros.  First H1(z) is designed to give the desired stopband
* response.  Then H0(z) is designed to force zeros into the overall
* impulse response.  The interaction between H0(z) and H1(z) is handled
* by iterating the design.  The stopband weighting for H1(z) is
* computed from the response of H0(z) in the stopband region.  In this
* way, the stopband of the overall Nyquist filter has the weighted
* equiripple behaviour specified by the input weighting.
      DO 600 ITER=1,NIMAX

* Update the weights to include the effect of H0(z)
        DO 300 I=1,NGRID
          WTH(I)=ABS(AMSFIR(FGRID(I),H0(L0+1),L0+1)) * WT(I)
 300    CONTINUE

* Find the extremal values for the stopband design
        CALL REMEZ(NGRID,XGRID,DES,WTH,DLIML,DLIMU,
     -             X,Y,LIM,NEXT,NCOF,DEVS,IER)

* Check for convergence (extremal values unchanged).  The
* extremal values lie on a finite grid.  Unchanged extremal
* values do not imply unchanged coefficient values.  However,
* the coefficient values should have more or less settled
* down when the extremal values stop changing.
        CONVRG=.TRUE.
        DO 400 I=1,NEXT
          IF (X(I).NE.XS(I)) CONVRG=.FALSE.
          XS(I)=X(I)
 400    CONTINUE
        IF (CONVRG) GO TO 700

* Find the coefficients of the approximation
        NCOFS=MIN(NCOF,NGRID)
        CALL REMCOF(XGRID(1),XGRID(NGRID),NEXT,X,Y,NCOFS,ALPHA)
        DO 420 I=NCOFS+1,NCOF
          ALPHA(I)=0D0
 420    CONTINUE

* Calculate the impulse response H1(z)
        CALL IMPULS(ICASE,ALPHA,H1,NH1)

* Solve for the coefficients of H0(z) which force the Nyquist response
        CALL NYQSLV(H1(L1+1),L1+1,NN,H0(L0+1),L0+1)
        DO 520 I=1,L0
          H0(I)=H0(NH0+1-I)
 520    CONTINUE

 600  CONTINUE
      CALL WARN('NYQFLT - Too many iterations')
      IER=3

* Generate the Nyquist filter (unscaled)
 700  CALL DCONLP(H,H0,NH0,H1,NH1)

* Set the scaling so that the overall Nyquist filter has a middle
* sample with value 1/NN.  This scaling corresponds to an approximately
* unity gain in the passband.
      NC=L1+L0+1
      DO 840 I=1,NH
        H(I)=H(I)/H(NC)
 840  CONTINUE

* Force exact zero crossings
      DO 860 I=NC+NN,NH,NN
        H(I)=0D0
        H(NH-I+1)=0D0
 860  CONTINUE
      H(NC)=1D0


      RETURN

      END
