*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE NYQDSG (H, NH, NN, NF, FRSTP, WTSTP,
*                        GRIDD, DEVS, IER)
*
*
* Purpose:
*     Design a Nyquist filter
*
*
* Description:
*     This routine designs a linear phase FIR Nyquist filter.  The
*     design algorithm is based on that described in the reference but
*     the present design does not force the filter to be factorable.
*     Reference:
*       R.P. Ramachandran and P. Kabal, "Minimax design of factorable
*       Nyquist filters for data transmission systems", Signal
*       Processing, November 1989.
*
*
* Parameters:
* R <-  H      - Array of resultant filter coefficients for the
*                Nyquist filter
* I ->  NH     - Number of filter coefficients desired (odd, maximum
*                255)
* I ->  NN     - Zero crossing interval
* I ->  NF     - Number of stopband points
* R ->  FRSTP  - Array of normalized stopband frequencies. These values
*                must be in increasing order.
* R ->  WTSTP  - Array of desired stopband weightings
* R ->  GRIDD  - Grid density.  The number of points in the frequency
*                grid used to determine the extrema of the
*                approximating function is the product of GRIDD and the
*                number of degrees of freedom for the stopband
*                response (approximately NH*(NN-1)/(2*NN)).
*                 0 - If GRIDD is zero, the grid density is chosen to
*                     be near the largest possible (given the internal
*                     array space available) up to a maximum of 16
*                -1 - If GRIDD is -1, the grid density is chosen to be
*                     near the largest possible given the internal
*                     array space available
* R <-  DEVS   - Resultant stopband deviation from the desired
*                specifications.  This value is unweighted.  The
*                deviation at a given frequency is obtained by dividing
*                by the weight at that frequency.
* I <-  IER    - Error parameter coded as follows.
*                0  - No error
*                1  - Remez algorithm failed to converge, the deviation
*                     is not monotonically increasing
*                2  - Remez algorithm failed to converge, too many
*                     iterations
*                3  - Too many iterations in the Nyquist design
*                For these cases, this routine returns coefficient
*                values.  For all other errors, an error message is
*                printed and execution is halted.
*
*
* Routines required:
*     FIRCHG - Find FIR filter specifications on a frequency grid
*     HALT   - Print an error message, stop with error status set
*     NYQFLT - Design a Nyquist filter
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:24:23 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE NYQDSG (H, NH, NN, NF, FRSTP, WTSTP,
     -                   GRIDD, DEVS, IER)


      INTEGER BPF,NGMAX,NHMAX,MXPT
      PARAMETER (BPF=1)
      PARAMETER (NGMAX=1000,NHMAX=255,MXPT=201)

      INTEGER NN,NF,IER
      INTEGER NBANDS,I,NH,L0,L1,NH1,ITYPE,ICASE,NGRID
      INTEGER IBAND(2)

      REAL H(NH),FRSTP(*),WTSTP(*),GRIDD,DEVS
      REAL FREQ(MXPT),WEIGHT(MXPT),VAL(MXPT),VLIML(MXPT),VLIMU(MXPT)
      REAL FGRID(NGMAX),XGRID(NGMAX),DES(NGMAX),WT(NGMAX),
     -     DLIML(NGMAX),DLIMU(NGMAX)

      DOUBLE PRECISION HD(NHMAX)


* Add a passband value
      IBAND(1)=1
      FREQ(1)=0.0
      VAL(1)=1.0
      WEIGHT(1)=1.0
      VLIML(1)=1.0
      VLIMU(1)=1.0

* Fill in the stopband values
      IBAND(2)=NF+1
      NBANDS=2
      DO 100 I=2,NF+1
        FREQ(I)=FRSTP(I-1)
        VAL(I)=0.0
        WEIGHT(I)=WTSTP(I-1)
        VLIML(I)=-1E20
        VLIMU(I)=+1E20
 100  CONTINUE

* Filter length
      IF (MOD(NH,2).EQ.0)
     -  CALL HALT('NYQDSG - Number of coefficients must be odd')
      L0=(NH-1)/(2*NN)
      L1=(NH-1)/2-L0
      NH1=2*L1+1

* Convert to a specification on a dense grid of frequencies
      ITYPE=BPF
      CALL FIRCHG(NH1,ITYPE,GRIDD,IBAND,NBANDS,FREQ,VAL,WEIGHT,
     -            VLIML,VLIMU,NGMAX,ICASE,NGRID,FGRID,XGRID,
     -            DES,WT,DLIML,DLIMU)

* Design the filter
      CALL NYQFLT(NGRID,FGRID,XGRID,DES,WT,DLIML,DLIMU,
     -            HD,NH,NN,DEVS,IER)

* Copy the filter coefficients into the output array
      DO 800 I=1,NH
        H(I)=HD(I)
 800  CONTINUE


      RETURN

      END
