*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE MPHFCT (NGRID, FGRID, XGRID, DES, WT, DLIML, DLIMU,
*                        HM, NHM, NN, DEVS, IER)
*
*
* Purpose:
*     Design a Nyquist filter (minimum phase factor)
*
*
* Description:
*     This routine designs a factorable linear phase FIR Nyquist
*     filter.  The Nyquist filter is factored into minimum and maximum
*     phase components.  The minimum phase factor is generated by this
*     routine.
*
*
* Parameters:
* I ->  NGRID  - Number of elements in each of FGRID, XGRID, DES, WT,
*                DLIML, and DLIMU (maximum 1000)
* R ->  FGRID  - Array of normalized frequency values.  These values
*                values must be in increasing order in the range
*                [0,1/2].
* R ->  XGRID  - Array of abscissa values (grid points).  These should
*                be in decreasing order in the range [-1,+1].  The
*                element XGRID(i) is assumed to be equal to
*                COS(2*Pi*FGRID(i)).
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid,
*                DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid,
*                DLIMU(i) >= DES(i)
* R <-  HM     - Array of resultant filter coefficients (double
*                precision) for the minimum phase factor of a Nyquist
*                filter
* I ->  NHM    - Number of filter coefficients desired (maximum 128)
* I ->  NN     - Zero crossing interval
* R <-  DEVS   - Resultant stopband deviation from the desired
*                specifications.  This value is unweighted.  The
*                deviation at a given frequency is obtained by dividing
*                by the weight at that frequency.
* I <-  IER    - Error parameter coded as follows.
*                0  - No error
*                3  - Too many iterations in the Nyquist design
*                For these cases, this routine returns coefficient
*                values.  For all other errors, an error message is
*                printed and execution is halted.
*
*
* Routines required:
*     AMSFIR - Find the amplitude response of a symmetric FIR filter
*     DCONLP - Convolve symmetric responses
*     HALT   - Print an error message, stop with error status set
*     IMPULS - Find the impulse response for an FIR filter
*     MINPHS - Find a minimum phase factor of a Nyquist filter
*     NYQSLV - Solve for a cascade Nyquist filter
*     REMCOF - Generate the sum of cosines coefficients for a filter
*     REMEZ  - Remez exchange algorithm for filter design
*     WARN   - Print a warning message on the standard error unit
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:24:25 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE MPHFCT (NGRID, FGRID, XGRID, DES, WT, DLIML, DLIMU,
     -                   HM, NHM, NN, DEVS, IER)


      INTEGER NGMAX,NHMMAX,NHMAX,L0MAX,L1MAX,NH1MAX,
     -        NCFMAX,MAXEXT,NIMAX
      PARAMETER (NGMAX=1000,NHMMAX=128,NHMAX=2*NHMMAX-1,
     -           L0MAX=(NHMAX-1)/2,L1MAX=(NHMAX-1)/2,
     -           NH1MAX=L1MAX+1,
     -           NCFMAX=(NH1MAX+1)/2,MAXEXT=NCFMAX+1,NIMAX=10)

      INTEGER NGRID,NHM,NN,IER
      INTEGER NH,L0,L1,NH0,NH1,NH2,ICASE,NCOF,I,NEXT,ITER,
     -        NCOFS,NH0M
      INTEGER LIM(MAXEXT)

      LOGICAL CONVRG

      REAL FGRID(NGRID),XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID),DEVS
      REAL WTH(NGMAX)

      DOUBLE PRECISION HM(NHM)
      DOUBLE PRECISION ALPHA(NCFMAX),X(MAXEXT),XS(MAXEXT),
     -     Y(MAXEXT),H0(2*L0MAX+1),H1(NH1MAX),H0M(2*L0MAX+1),
     -     H2(2*L1MAX+1)
      DOUBLE PRECISION SS,SF
      DOUBLE PRECISION AMSFIR


* Array sizes;
*   H(z) = H0(z) H2(z),    where H2(z) = H1(z) H1(z)
*   Hm(z) = H0m(z) H1(z),  where H0(z) = H0m(z) H0m(1/z).
*
*   NH=2*NHm-1, NH2=2*NH1-1, NHm=NH0m+NH1-1, NH0=2*NH0m-1
*
*   L0=(NH-1)/(2*NN), L1=(NH-1)/2 - L0
*   Largest L0 occurs for NN=1, L0max=(NHmax-1)/2, where
*   NHmax=2*NHmmax-1.
*   Largest L1 occurs for NN large, L1max=(NHmax-1)/2.
*
*   In terms of L0 and L1,
*   NH1=L1+1, NH2=2*L1+1, NH0=2*L0+1, NH0m=L0+1, NHm=L0+L1+1
*
*   Largest array dimensions,
*   NH1max=L1max+1, NH2max=2*L0max+1, NH0max=2*L0max+1,
*   NH0mmax=L0max+1, NHmmax is a given quantity.
*   To prevent problems lest the minimum phase factorization go
*   awry and return too many terms, NH0mmax is set equal to NH0max.

* Filter lengths
      NH=2*NHM-1
      L0=(NH-1)/(2*NN)
      L1=(NH-1)/2-L0
      NH0=2*L0+1
      NH1=L1+1
      NH2=2*L1+1
      IF (MOD(NH1,2).NE.0) THEN
        ICASE=1
        NCOF=(NH1-1)/2+1
      ELSE
        ICASE=2
        NCOF=NH1/2
      END IF

* Initialization
      DO 200 I=1,NH0
        H0(I)=0D0
 200  CONTINUE
      H0(L0+1)=1D0

      NEXT=MIN(NCOF+1,NGRID)
      DO 220 I=1,NEXT
        XS(I)=0D0
 220  CONTINUE

* The Nyquist filter is expressed as H0(z) H1(z) H1(z).  H1(z) is a
* filter with all of its zeros in the stopband.  H0(z) has the pass-
* band zeros.  First H1(z) is designed to give the desired stopband
* response.  Then H0(z) is designed to force zeros into the overall
* impulse response.  The interaction between H0(z) and H1(z) is handled
* by iterating the design.  The stopband weighting for H1(z) is
* computed from the response of H0(z) in the stopband region.  In this
* way, the stopband of the overall Nyquist filter has the weighted
* equiripple behaviour specified by the input weighting.
      DO 600 ITER=1,NIMAX

* Update the weights to include the effect of H0(z)
        DO 300 I=1,NGRID
          WTH(I)=SQRT(ABS(AMSFIR(FGRID(I),H0(L0+1),L0+1))) * WT(I)
 300    CONTINUE

* Find the extremal values for the stopband design
        CALL REMEZ(NGRID,XGRID,DES,WTH,DLIML,DLIMU,
     -             X,Y,LIM,NEXT,NCOF,DEVS,IER)

* Check for convergence (extremal values unchanged).  The
* extremal values lie on a finite grid.  Unchanged extremal
* values do not imply unchanged coefficient values.  However,
* the coefficient values should have more or less settled
* down when the extremal values stop changing.
        CONVRG=.TRUE.
        DO 400 I=1,NEXT
          IF (X(I).NE.XS(I)) CONVRG=.FALSE.
          XS(I)=X(I)
 400    CONTINUE
        IF (CONVRG) GO TO 700

* Find the coefficients of the approximation
        NCOFS=MIN(NCOF,NGRID)
        CALL REMCOF(XGRID(1),XGRID(NGRID),NEXT,X,Y,NCOFS,ALPHA)
        DO 420 I=NCOFS+1,NCOF
          ALPHA(I)=0D0
 420    CONTINUE

* Calculate the impulse response H1(z)
        CALL IMPULS(ICASE,ALPHA,H1,NH1)

* Convolve the coefficients of H1(z) with themselves
* Solve for the coefficients of H0(z) which force the Nyquist response
        CALL DCONLP(H2,H1,NH1,H1,NH1)
        CALL NYQSLV(H2(L1+1),L1+1,NN,H0(L0+1),L0+1)
        DO 520 I=1,L0
          H0(I)=H0(NH0+1-I)
 520    CONTINUE

 600  CONTINUE
      CALL WARN('MPHFCT - Too many iterations')
      IER=3

* Find the minimum phase factor of H0(z)
* This factor is unscaled
 700  CALL MINPHS(H0,NH0,H0M,NH0M)
      IF (2*NH0M-1.NE.NH0)
     -  CALL HALT('MPHFCT - Invalid minimum phase factorization')

* Generate the minimum phase factor of the Nyquist filter (unscaled)
      CALL DCONLP(HM,H0M,NH0M,H1,NH1)

* Set the scaling so that the overall Nyquist filter has a middle
* sample with value 1/NN.  This scaling corresponds to an approximately
* unity gain in the passband.
* The overall Nyquist filter is obtained as Hm(z) Hm(1/z).  The middle
* sample of the Nyquist response is the sum of the squares of the
* coefficients of the minimum phase filter.
      SS=0D0
      DO 820 I=NHM,1,-1
        SS=SS + HM(I)**2
 820  CONTINUE
      SF=1D0/SQRT(SS*NN)
      DO 840 I=1,NHM
        HM(I)=SF*HM(I)
 840  CONTINUE


      RETURN

      END
