*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE MINPHS (H, NH, HMIN, NHMIN)
*
*
* Purpose:
*     Find a minimum phase factor of a Nyquist filter
*
*
* Description:
*     This routine finds the roots of the given response polynomial.
*     The response is considered to be a polynomial in 1/z.  Consider
*     the roots in the z-plane.  This routine returns a polynomial
*     formed as the product of the root factors corresponding to roots
*     on or inside the unit circle in the z-plane.  The input polyomial
*     is
*
*              NH-1
*       H(z) = SUM h(n) (1/z)**n .
*              n=0
*
*     The output polynomial is
*
*                NHmin-1
*       Hmin(z) = PROD  (1 - zr(n)/z) ,
*                 n=1
*
*     where zr(n) is a z-plane root of H(z) with magnitude less than
*     or equal to unity.
*
*
* Parameters:
* D ->  H      - Array of input double precision response coefficients
*                corresponding to a polynomial in 1/z.
* I ->  NH     - Number of response coefficients (maximum 128)
* D <-  HMIN   - Output array of double precision response coefficients
*                corresponding to the minimum phase response.  This is
*                a monic polynomial (leading coefficient is unity).  If
*                H(n) has trailing zero coefficients, an equal number
*                of trailing coefficients are set in HMIN.  These zero
*                valued coefficients correspond to z-plane roots at
*                zero.
* I <-  NHMIN  - Number of response coefficients in HMIN (up to NH
*                values).
*
*
* Routines required:
*     DCONLP - Convolve symmetric responses
*     DRPOLY - Find the roots of a polynomial with real coefficients
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:21:37 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE MINPHS (H, NH, HMIN, NHMIN)


      INTEGER NHMAX
      PARAMETER (NHMAX=128)

      INTEGER NH,NHMIN
      INTEGER NROOT,I,J,NXP,NHH
      INTEGER INDR(NHMAX-1)

      DOUBLE PRECISION H(0:NH-1),HMIN(*)
      DOUBLE PRECISION XP(3),TEMP(NHMAX),HH(NHMAX),
     -     ZR(NHMAX-1),ZI(NHMAX-1)

      DATA XP(1)/1D0/


* Find the roots of the polynomial in p=1/z.
* The polynomial H(p) = h(0) + h(1)*p + h(2)*(p**2) + ... .
* The number of roots is NROOT <= NH-1.  If NROOT is less than NH-1,
* the remaining roots are at p=infinity
      CALL DRPOLY(H,NH,ZR,ZI,NROOT)

* Initialization
      DO 120 I=1,NROOT
        INDR(I)=1
 120  CONTINUE
      NHH=1
      HH(1)=1D0

* Look for roots of the polynomial in p which are on or outside
* the unit circle.  In terms of the polynomial in z, these are
* the roots in the z-plane which lie on or inside the unit circle).
      DO 400 I=1,NROOT

        IF (INDR(I).NE.0 .AND.
     -    (ZR(I)**2 + ZI(I)**2).GE.1D0) THEN

* Search for the conjugate root (real roots will be paired
* with themselves)
          DO 220 J=I,NROOT
            IF (ZR(I).EQ.ZR(J) .AND. ZI(I).EQ.-ZI(J)) THEN
              INDR(J)=0
              GO TO 300
            END IF
 220      CONTINUE
          CALL HALT('MINPHS - Roots not in conjugate pairs')

* Form the product of the root factor polynomials, using the reciprocal
* of the p-plane roots.
*   NRoot
*   PROD (1 - p / pr(i))
*    i=1
 300      IF (I.EQ.J) THEN
            XP(2)=-1D0/ZR(I)
            NXP=2
          ELSE
            XP(2)=-2D0*ZR(I)/(ZR(I)**2 + ZI(I)**2)
            XP(3)=1D0/(ZR(I)**2 + ZI(I)**2)
            NXP=3
          END IF
          CALL DCONLP(TEMP,XP,NXP,HH,NHH)
          NHH=NHH+NXP-1
          DO 340 J=1,NHH
            HH(J)=TEMP(J)
 340      CONTINUE
        END IF

 400  CONTINUE

* Copy the result into the output array, append zeros if necessary
* to account for the roots at z=0 (p=infinity).
      DO 520 J=1,NHH
        HMIN(J)=HH(J)
 520  CONTINUE
      NHMIN=NHH+NH-NROOT-1
      DO 540 J=NHH+1,NHMIN
        HMIN(J)=0D0
 540  CONTINUE


      RETURN

      END
