*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     REAL FUNCTION DMCINM (DXL, DXR, DYL, DYR)
*
*
* Purpose:
*     Calculate a derivative for a cubic at a middle point
*
*
* Description:
*     This routine calculates a derivative value that can be used to
*     generate piecewise monotone cubic hermite interpolants.  Consider
*     three points (xl,yl), (x0,y0) and (xr,yr), where xl < x0 < xr.
*     The derivative is calculated at the point (x0,y0).
*
*
* Parameters:
* R <-  DMCINM - Output derivative value at the reference point
*
* R ->  DXL    - Increment in abscissa value to the left of the
*                reference point (x0-xl, must be positive)
* R ->  DXR    - Increment in abscissa value to the right of the
*                reference point (xr-x0, must be positive)
* R ->  DYL    - Increment in ordinate value to the left (y0-yl)
* R ->  DYR    - Increment in ordinate value to the right (yr-y0)
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/24 20:44:53 $
*
*
*-----------------------------------------------------------------------

      REAL FUNCTION DMCINM (DXL, DXR, DYL, DYR)


      REAL DXL,DXR,DYL,DYR
      REAL SL,SR,SMAX,SMIN


* Check for increasing X values
      IF (DXL.LE.0.0 .OR. DXR.LE.0.0)
     -  CALL HALT('DMCINM - Abscissa values not in increasing order')


* SL is the slope to the left of point (x0,y0)
* SR is the slope to the right of point (x0,y0)
      SL=DYL/DXL
      SR=DYR/DXR

* Derivative is set to zero if slope changes sign
      IF ((SL.GT.0.0 .AND. SR.GT.0.0) .OR.
     -    (SL.LT.0.0 .AND. SR.LT.0.0)) THEN

* Slopes of the same sign
* Use Brodlie modification of Butland formula
*
*            Sl Sr                  1       xl-x0
*   d = --------------- , where a = - (1 + -------) .
*       a Sl + (1-a) Sr             3      xl - xr
*
* This formula gives the harmonic mean of the slopes when
* x0-xl=xr-x0. The factor a modifies the formula to take into
* account the relative spacing of the values (a varies from
* 1/3 to 2/3). This formula satisfies the properties
* (a) the formula is invariant to an exchange of the
*     the left and right points,
* (b) min(Sl,Sr) <= d <= max(Sl,Sr),
* (c) |d| <= 3 min(|Sl|,|Sr|). This is a sufficient condition
*     for monotonicity if d is calculated in this way for
*     neighboring points.
*
* The following calculation is designed to avoid overflows, but
* is equivalent to the above formula when Sl and Sr have the
* same sign.
        SMAX=MAX(ABS(SL),ABS(SR))
        SMIN=MIN(ABS(SL),ABS(SR))
        DMCINM=SMIN/( (DXL+DXR+DXR)/(3.*(DXL+DXR))*(SL/SMAX)
     -              + (DXL+DXR+DXL)/(3.*(DXL+DXR))*(SR/SMAX) )

* Slopes of opposite signs - force the derivative to zero to
* give a local maximum at the middle data point
      ELSE
        DMCINM=0.0

      END IF


      RETURN

      END
