*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE INTDSG (H, NTAP, IR, F, PSD, NVAL, ERR2, IER)
*
*
* Purpose:
*     Design a min. mean-square error interpolating filter
*
*
* Description:
*     This subroutine calculates the coefficients for an interpolating
*     filter which minimizes the mean-square interpolation error given
*     the power spectrum of the input signal.  The power spectrum of
*     the input signal is specified in terms of a continuous component
*     and sinusoidal components.  The continuous component is given as
*     a tabulated function.  A piecewise monotonic cubic interpolation
*     is used between the given values.  The sinusoidal components are
*     specified by frequency and power.  The continuous power spectrum
*     is given as pairs of values: a normalized frequency and the
*     corresponding power spectrum value.  The frequencies must be
*     non-negative and be in increasing order.  The continuous power
*     spectrum is assumed to be zero outside the given range of
*     frequencies.  The cubic interpolant has zero slope at the first
*     and last points of the continuous portion of the power spectrum.
*     Sinusoidal components of the power spectrum are specified as a
*     normalized frequency (non-negative) and the corresponding power.
*
*     Conceptually, interpolation has two steps.  In the first, the
*     sampling rate of the input sequence is increased by a factor
*     of IR, by inserting IR-1 zeros between each input sample.  In
*     the second step, the increased rate sequence is applied to the
*     interpolating filter.
*
*     Reference: G. Oetken, T.W. Parks and H.W. Schussler,
*                "New results in the design of digital interpolators",
*                IEEE ASSP, pp. 301-309, June 1975
*
*
* Parameters:
* R <-  H      - Resultant vector of filter coefficients
* I ->  NTAP   - Number of filter coefficients (maximum 42*IR+1).
* I ->  IR     - Interpolation ratio
* R ->  F      - Array of normalized frequencies.  The frequency values
*                must be non-negative.  The normalizing frequency is the
*                frequency of the input sequence before interpolation.
*                The first NVAL(1) values must be in increasing order.
*                These are the frequencies of samples of the continuous
*                power spectral density.  The next NVAL(2) values are
*                the frequencies of the sinusoidal components.
* R ->  PSD    - Array of power spectrum values. The value PSD(i) gives
*                the power spectrum at frequency F(i). Each PSD(i) must
*                be non-negative.  The first NVAL(1) values are samples
*                of the continuous power spectral density.  The next
*                NVAL(2) values are the powers of the sinusoidal
*                components.
* I ->  NVAL   - Number of frequency and power spectrum values.  This
*                is a two element array.  NVAL(1) gives the number of
*                samples of the continuous power spectrum (at least 2).
*                NVAL(2) gives the number of sinusoidal components.
* R <-  ERR2   - Resulting normalized mean-square interpolation error.
*                The interpolation error depends on the relative phase
*                of the sample being interpolated.  ERR2 is the average
*                of the normalized mean-square errors for each possible
*                sample phase.
* I <-  IER    - Error code,
*                  0 - No error
*                  1 - Matrix numerically not positive definite. Filter
*                      coefficients are not returned in this case.
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     ICEIL  - Calculate the ceiling function of a ratio of integers
*     IFLOOR - Calculate the floor function of a ratio of integers
*     SNRINT - Calculate the mean-square interpolation error
*     TACORR - Evaluate a correlation, given spectrum values
*     TPLSLV - Solve a Toeplitz set of equations
*     WARN   - Print a warning message on the standard error unit
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:20:32 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE INTDSG (H, NTAP, IR, F, PSD, NVAL, ERR2, IER)


      INTEGER NMAX
      PARAMETER (NMAX=42)

      INTEGER NTAP,IR,NVAL(2),IER
      INTEGER IS,NR,NT,NH,IVL,IVR,IUL,IUR
      INTEGER NM,I,LL,LU,J,NN,M,K
      INTEGER IC
      INTEGER MODP,N1,N2
      INTEGER IFLOOR,ICEIL

      REAL H(0:NTAP-1),F(*),PSD(*),ERR2
      REAL SNRINT

      DOUBLE PRECISION RXX(0:NMAX-1),P(0:NMAX-1),HS(0:NMAX-1)
      DOUBLE PRECISION TACORR
      DOUBLE PRECISION R


* Statement function to calculate the correlation function (integer
* argument)
      R(IC) = TACORR(DBLE(IC)/NR,F,PSD,NVAL)

* Statement function to calculate the modulo function with a positive
* remainder (N1 of any sign, N2 positive)
      MODP(N1,N2)=N1-N2*IFLOOR(N1,N2)


* Let the number of taps be odd, Nt=2*Nh+1.
* The interpolating filter is divided into NR subfilters,
*   hs(m,i) = h(m*NR+i) , where 0 <= i < NR

* The equations to be solved for the i'th subfilter are
*   R hs[i] = p[i] ,
* where the matrix R is Toeplitz with elements R(m,k) = r((m-k)*NR)
*       the vector hs[i] has elements hs(m,i), and
*       the vector p[i] has elements p(m,i) = r(m*NR + i)

* The case of an even number of coefficients is handled by conceptually
* creating an interpolating filter with an interpolating ratio of 2*IR.
* This double length filter (adding one coefficient between every one
* of the original filter) has an odd length.  Only coefficients for
* every second subfilter are calculated.

* NR  - Effective interpolating ratio
* IS  - Subsampling factor, 1 for odd length filters
*                           2 for even length filters
* NT  - Number of taps in the filter with interpolating ratio NR,
*       (IS-1) extra coefficients between each real coefficient

      IF (MOD(NTAP,2).EQ.0) THEN
        IS=2
      ELSE
        IS=1
      END IF
      NR=IS*IR
      NT=NTAP+(IS-1)*(NTAP-1)

* Calculate the starting and ending coefficients
* The filter has coefficient indices -NH,...,-1,0,+1,...,+NH
*   -NH = IUL*NR + IVL, where 0 <= IVL < NR
*   +NH = IUR*NR + IVR, where 0 <= IVR < NR
      NH=(NT-1)/2
      IVL=MODP(-NH,NR)
      IVR=MODP(NH,NR)
      IUL=IFLOOR(-NH-IVL,NR)
      IUR=IFLOOR(NH-IVR,NR)

* Find the length of the longest subfilter
* (1) if the interpolating ratio is one, the solution is the trivial
*     filter with subfilter index 0
* (2) if NTAP is of the form 2*IR*k+1, the filter will have zeros at
*     each end and the effective length can be decreased by two
* (3) in other cases the first coefficient will be belong to a
*     a subfilter of maximum length (other subfilters may be as long)
      IF (NR.EQ.1) THEN
        NM=0
      ELSE IF (MOD(NTAP-1,2*IR).EQ.0) THEN
        NM=ICEIL(NTAP-2,IR)
      ELSE
        NM=ICEIL(NTAP,IR)
      END IF
      IF (NM.GT.NMAX)
     -  CALL HALT('INTDSG - Too many coefficients')

* Fill in the autocorrelation vector
      DO 120 M=0,NM-1
        RXX(M)=R(M*NR)
 120  CONTINUE

* The interpolating filter is symmetrical,
*   h(n) = h(-n) , where -Nh <= n <= +Nh.
* The symmetry of the overall filter forces a time-reverse symmetry on
* the subfilters,
*   hs(m,i) = hs(-m,NR-i) .
* In addition, the subfilter for i=0 has elements [0,...0,1,0,...,0].
* This means that only the subfilters for i=1,...,NR/2 need to be
* explicitly calculated.

      DO 400 I=1,NR/2,IS

* Calculate the length of the I'th subfilter
        LL=IUL
        IF (I.LT.IVL) LL=LL+1
        LU=IUR
        IF (I.GT.IVR) LU=LU-1

* Fill in the right-hand side vector, p[m]
        J=LL*NR+I
        NN=LU-LL+1
        DO 320 M=0,NN-1
          P(M)=R(J)
          J=J+NR
 320    CONTINUE

* Find the coefficients of the subfilter, solving A hs[m] = p[m]
        CALL TPLSLV(RXX,P,NN,HS,IER)
        IF (IER.NE.0) THEN
          CALL WARN('INTDSG - Non-positive definite equations')
          GO TO 900
        END IF

* Copy the subfilter coefficients into the final filter:
* Interpolating filter, interpolating ratio NR
*  index j: -NH, ... , -1, 0, +1, ... , +NH
* Interpolating filter, interpolating ratio IR
*  index k: 0, 1, ... , NTAP-1
* The mapping between these is k=(j+NH)/IS
        J=LL*NR+I
        K=(J+NH)/IS
        DO 340 M=0,NN-1
          H(K)=HS(M)
          H(NTAP-1-K)=HS(M)
          K=K+IR
 340    CONTINUE

 400  CONTINUE

* Fill in the known values of H (for NTAP odd)
      IF (MOD(NTAP,2).NE.0) THEN
        I=0
        LL=IUL
        IF (I.LT.IVL) LL=LL+1
        LU=IUR
        IF (I.GT.IVR) LU=LU-1
        J=LL*NR+I
        K=(J+NH)/IS
        DO 500 M=LL,LU
          IF (M.EQ.0) THEN
            H(K)=1.
          ELSE
            H(K)=0.
          END IF
          K=K+IR
 500    CONTINUE
      END IF

* Calculate the average mean-square error
      ERR2=SNRINT(H,NTAP,IR,F,PSD,NVAL)


 900  RETURN

      END
