*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE GRIDPT (ICASE, NGRIDD, IBAND, NBANDS,
*                        FREQ, VAL, WEIGHT, VLIML, VLIMU,
*                        NGMAX, NGRID, FGRID, DES, WT, DLIML, DLIMU)
*
*
* Purpose:
*     Find the filter specifications on a frequency grid
*
*
* Description:
*     Given the input band specifications, this routine generates an
*     equivalent specification on a dense array of frequency points.
*     The intermediate values are generated by interpolating using a
*     piecewise monotonic cubic function between the values of the
*     input specifications.
*
*
* Parameters:
* I ->  ICASE  - Type of filter
*                1 - even number of coefficients, bandpass filter
*                2 - odd number of coefficients, bandpass filter
*                3 - even number of coefficients, differentiator or
*                    Hilbert transform
*                4 - odd number of coefficients, differentiator or
*                    Hilbert transform
* I ->  NGRIDD - Desired number of grid points.  The actual number of
*                grid points is determined by this routine to be a
*                value less than or equal to NGRIDD.
* I ->  IBAND  - Frequency band specification index.  The array FREQ is
*                used to specify the frequency bands.
*                 Band    Lower Band Edge          Upper Band Edge
*                  1     FREQ(1)                  FREQ(IBAND(1))
*                  2     FREQ(IBAND(1)+1)         FREQ(IBAND(2))
*                 ...        ...                    ...
*                NBANDS  FREQ(IBAND(NBANDS-1)+1)  FREQ(IBAND(NBANDS))
*                VAL(i), WEIGHT(i), VLIML(i), and VLIMU(i) specify the
*                desired value, weight, lower limit and upper limit at
*                frequency FREQ(i).  The total number of frequencies
*                specified (IBAND(NBANDS)) must be less than 1000.
* I ->  NBANDS - Number of bands specified
* R ->  FREQ   - Array of normalized frequencies.  These values must be
*                in increasing order.
* R ->  VAL    - Array of desired values
* R ->  WEIGHT - Array of weights
* R ->  VLIML  - Array of lower constraints
* R ->  VLIMU  - Array of upper constraints
* I ->  NGMAX  - Largest permissible value for NGRID
* I <-  NGRID  - Number of points in the frequency grid
* R <-  FGRID  - Array specifying the normalized frequency values
* R <-  DES    - Array of desired values on the frequency grid
* R <-  WT     - Array of desired weights on the frequency grid
* R <-  DLIML  - Array of lower constraints
* R <-  DLIMU  - Array of upper constraints
*
*
* Routines required:
*     BANDPT - Determine the specifications for frequency bands
*     HALT   - Print an error message, stop with error status set
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:00:12 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE GRIDPT (ICASE, NGRIDD, IBAND, NBANDS,
     -                   FREQ, VAL, WEIGHT, VLIML, VLIMU,
     -                   NGMAX, NGRID, FGRID, DES, WT, DLIML, DLIMU)


      INTEGER BPF,REC,DIF,HIL
      PARAMETER (BPF=1,REC=2,DIF=3,HIL=4)

      INTEGER ICASE,NGRIDD,NBANDS,IBAND(NBANDS),NGMAX,NGRID
      INTEGER IU,I,IL,N,NVAL,MAXNPT,NPT

      REAL FREQ(*),VAL(*),WEIGHT(*),VLIML(*),VLIMU(*),
     -     FGRID(NGMAX),DES(NGMAX),WT(NGMAX),
     -     DLIML(NGMAX),DLIMU(NGMAX)
      REAL FS,DELF,FMIN,FMAX


* Find the grid spacing.
* FS is the frequency extent of the specified bands.
* The maximum frequency increment DELF is determined based on
* the fact that a sample occurs at each specified point and
* additional samples are uniformly placed between each pair
* of points such that the maximum spacing is at most DELF.
      FS=0.0
      IU=0
      DO 120 I=1,NBANDS
        IL=IU+1
        IU=IBAND(I)
        FS=FS+(FREQ(IU)-FREQ(IL))
 120  CONTINUE
      DELF=FS/MIN(NGMAX-IBAND(NBANDS),NGRIDD)

* For cases 2 and 3 problems occur at the maximum frequency
* (desired value divided by zero but with zero weight), while
* for cases 3 and 4 similar problems occur at zero frequency
      IF (ICASE.EQ.1) THEN
        FMIN=0.0
        FMAX=0.5
      ELSE IF (ICASE.EQ.2) THEN
        FMIN=0.0
        FMAX=0.5-DELF
      ELSE IF (ICASE.EQ.3) THEN
        FMIN=DELF
        FMAX=0.5-DELF
      ELSE
        FMIN=DELF
        FMAX=0.5
      END IF

*********************
* Calculate the desired response and the weight on the
* frequency grid.
* Sample points are uniformly spaced between the specified
* frequencies with a frequency separation which is at most DELF
      N=0
      IU=0
      DO 280 I=1,NBANDS
        IL=IU+1
        IU=IBAND(I)

        MAXNPT=NGMAX-N
        IF (MAXNPT.LE.0) CALL HALT('GRIDPT - Insufficient array '//
     -                            'space for the given grid density')
        NVAL=IU-IL+1
        CALL BANDPT(FREQ(IL),VAL(IL),WEIGHT(IL),VLIML(IL),VLIMU(IL),
     -              NVAL,FMIN,FMAX,DELF,MAXNPT,NPT,
     -              FGRID(N+1),DES(N+1),WT(N+1),
     -              DLIML(N+1),DLIMU(N+1))
        N=N+NPT

 280  CONTINUE

      NGRID=N


      RETURN

      END
