*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE FIRREM (NGRID, XGRID, DES, WT, DLIML, DLIMU,
*                        ICASE, H, NH, DEVS, IER)
*
*
* Purpose:
*     Design an FIR filter: Remez exchange algorithm
*
*
* Description:
*     This routine designs multiple band bandpass filters,
*     differentiators, or Hilbert transform filters using the Remez
*     exchange algorithm.
*     References:
*       J.H. McClellan, T.W. Parks and L.R. Rabiner, "A computer
*       program for designing optimum FIR linear phase digital
*       filters", IEEE Trans. Audio and Electroacoustics, vol. AU-21,
*       pp. 506-526, December 1973.
*       F. Grenez, "Design of linear or minimum-phase FIR filters by
*       constrained Chebyshev approximation", Signal Processing,
*       vol. 5, pp. 325-332, July 1983.
*
*
* Parameters:
* I ->  NGRID  - Number of elements in each of XGRID, DES, WT,
*                DLIML, and DLIMU
* R ->  XGRID  - Array of abscissa values (grid points).  These should
*                be in decreasing order in the range [-1,+1].
* R ->  DES    - Desired values on the grid
* R ->  WT     - Weight values on the grid, WT(i) > 0
* R ->  DLIML  - Lower constraint values on the grid,
*                DLIML(i) <= DES(i)
* R ->  DLIMU  - Upper constraint values on the grid,
*                DLIMU(i) >= DES(i)
* I ->  ICASE  - Type of filter
*                1 - even number of coefficients, bandpass filter
*                2 - odd number of coefficients, bandpass filter
*                3 - even number of coefficients, differentiator or
*                    Hilbert transform
*                4 - odd number of coefficients, differentiator or
*                    Hilbert transform
* R <-  H      - Array of resultant filter coefficients (double
*                precision)
* I ->  NH     - Number of filter coefficients desired (maximum 256)
* R <-  DEVS   - Resultant deviation from the desired specifications.
*                This value is unweighted.  The deviation at a given
*                frequency is obtained by dividing by the weight at
*                that frequency.
* I <-  IER    - Error parameter coded as follows.
*                0  - No error
*                1  - Remez algorithm failed to converge, the deviation
*                     is not monotonically increasing
*                2  - Remez algorithm failed to converge, too many
*                     iterations
*                For these cases, this routine returns coefficient
*                values.  For all other errors, an error message is
*                printed and execution is halted.
*
*
* Routines required:
*     IMPULS - Find the impulse response for an FIR filter
*     REMCOF - Generate the sum of cosines coefficients for a filter
*     REMEZ  - Remez exchange algorithm for filter design
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:17:28 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE FIRREM (NGRID, XGRID, DES, WT, DLIML, DLIMU,
     -                   ICASE, H, NH, DEVS, IER)


      INTEGER NHMAX,NCFMAX
      PARAMETER (NHMAX=256,NCFMAX=(NHMAX+1)/2)

      INTEGER NGRID,ICASE,NH,IER
      INTEGER NCOF,NEXT,NCOFS,I
      INTEGER LIM(NCFMAX+1)

      REAL XGRID(NGRID),DES(NGRID),WT(NGRID),
     -     DLIML(NGRID),DLIMU(NGRID)
      REAL DEVS

      DOUBLE PRECISION H(NH)
      DOUBLE PRECISION ALPHA(NCFMAX),X(NCFMAX+1),Y(NCFMAX+1)


* Determine the number of design coefficients
      IF (ICASE.EQ.1) THEN
        NCOF=(NH+1)/2
      ELSE IF (ICASE.EQ.2) THEN
        NCOF=NH/2
      ELSE IF (ICASE.EQ.3) THEN
        NCOF=(NH-1)/2
      ELSE
        NCOF=NH/2
      END IF

* Find the extremal values for the approximation
      CALL REMEZ(NGRID,XGRID,DES,WT,DLIML,DLIMU,
     -           X,Y,LIM,NEXT,NCOF,DEVS,IER)

* Find the coefficients of the approximation
      NCOFS=MIN(NCOF,NGRID)
      CALL REMCOF(XGRID(1),XGRID(NGRID),NEXT,X,Y,NCOFS,ALPHA)
      DO 220 I=NCOFS+1,NCOF
        ALPHA(I)=0D0
 220  CONTINUE

* Calculate the impulse response
      CALL IMPULS(ICASE,ALPHA,H,NH)


      RETURN

      END
