*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE FIRCHG (NTAP, ITYPE, GRIDD, IBAND, NBANDS,
*                        FREQ, VAL, WEIGHT, VLIML, VLIMU,
*                        NGMAX, ICASE, NGRID,
*                        FGRID, XGRID, DES, WT, DLIML, DLIMU)
*
*
* Purpose:
*     Find FIR filter specifications on a frequency grid
*
*
* Description:
*     This routine generates a specification for an FIR filter.  This
*     specification is in terms of a dense frequency grid of desired
*     values, weights and limits.
*
*
* Parameters:
* I ->  NTAP   - Number of filter coefficients
* I ->  ITYPE  - Filter type.
*                1 - Multiple passband/stopband filter
*                2 - Multiple passband/stopband filter
*                    (sin(X)/X compensation)
*                3 - Differentiator
*                4 - Hilbert transform filter
* R ->  GRIDD  - Grid density.  The number of points in the frequency
*                grid used to determine the extrema of the
*                approximating function is the product of GRIDD and
*                the number of degrees of freedom for the filter
*                coefficients (approximately NTAP/2).
*                 0 - If GRIDD is zero, the grid density is chosen to
*                     be near the largest possible (given the internal
*                     array space available) up to a maximum of 16
*                -1 - If GRIDD is -1, the grid density is chosen to be
*                     near the largest possible given the internal
*                     array space available
* I ->  IBAND  - Frequency band specification index.  The array FREQ is
*                used to specify the frequency bands.
*                 Band    Lower Band Edge          Upper Band Edge
*                  1     FREQ(1)                  FREQ(IBAND(1))
*                  2     FREQ(IBAND(1)+1)         FREQ(IBAND(2))
*                 ...        ...                    ...
*                NBANDS  FREQ(IBAND(NBANDS-1)+1)  FREQ(IBAND(NBANDS))
*                VAL(i), WEIGHT(i), VLIML(i), and VLIMU(i) specify the
*                desired value, weight, lower limit, and upper limit
*                at frequency FREQ(i).  These values are interpolated
*                using a piecewise monotonic cubic function between
*                frequencies within a band.
* I ->  NBANDS - Number of bands specified.
* R ->  FREQ   - Array of normalized frequencies.  These values must be
*                in increasing order, 0 <= FREQ(i) < FREQ(i+1) <= 1/2.
* R ->  VAL    - Array of desired amplitude values (desired slopes for
*                differentiators)
* R ->  WEIGHT - Array of desired weightings.  For differentiators,
*                the given weighting applies to the slope.  WEIGHT(i)
*                must be positive.
* R ->  VLIML  - Array of lower constraints. VLIML(i) must be less than
*                or equal to VAL(i).
* R ->  VLIMU  - Array of upper constraints.  VLIMU(i) must be greater
*                than or equal to VAL(i).
* I ->  NGMAX  - Largest permissible value for NGRID
* I <-  ICASE  - Type of filter
*                1 - even number of coefficients, bandpass filter
*                2 - odd number of coefficients, bandpass filter
*                3 - even number of coefficients, differentiator or
*                    Hilbert transform
*                4 - odd number of coefficients, differentiator or
*                    Hilbert transform
* I <-  NGRID  - Number of points in the frequency grid
* R <-  FGRID  - Array specifying the normalized frequency grid values.
*                The resulting values will be in increasing order,
*                0 <= FGRID(i) < FGRID(i+1) <= 1/2.
* R <-  XGRID  - Array specifying the transformed frequency values.
*                The resulting values will be in decreasing order,
*                1 <= XGRID(i) < XGRID(i+1) <= -1.  The element
*                XGRID(i) is equal to COS(2*Pi*FGRID(i)).
* R <-  DES    - Array of desired values on the frequency grid
* R <-  WT     - Array of weights on the frequency grid, WT(i) > 0
* R <-  DLIML  - Array of lower constraints, DLIML(i) <= DES(i)
* R <-  DLIMU  - Array of upper constraints, DLIMU(i) >= DES(i)
*
*
* Routines required:
*     CHSPEC - Modify the specifications for an FIR filter
*     GRIDPT - Find the filter specifications on a frequency grid
*     TRSPEC - Transform the FIR design problem into a canonical form
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:17:32 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE FIRCHG (NTAP, ITYPE, GRIDD, IBAND, NBANDS,
     -                   FREQ, VAL, WEIGHT, VLIML, VLIMU,
     -                   NGMAX, ICASE, NGRID,
     -                   FGRID, XGRID, DES, WT, DLIML, DLIMU)


* The number of elements used in each of the arrays FGRID, XGRID, DES,
* WT, DLIML, and DLIMU is approximately proportional to the grid
* density and to the number of coefficients.

      INTEGER BPF,REC,DIF,HIL
      REAL GRIDD0
      PARAMETER (BPF=1,REC=2,DIF=3,HIL=4)
      PARAMETER (GRIDD0=16.)

      INTEGER NTAP,ITYPE,NBANDS,IBAND(NBANDS),NGMAX,ICASE,NGRID
      INTEGER NCOF,NGRIDD

      REAL GRIDD,FREQ(*),VAL(*),WEIGHT(*),VLIML(*),VLIMU(*),
     -     FGRID(*),XGRID(*),DES(*),WT(*),DLIML(*),DLIMU(*)


* Determine the type of design
      IF (ITYPE.EQ.BPF .OR. ITYPE.EQ.REC) THEN
        IF (MOD(NTAP,2).NE.0) THEN
          ICASE=1
          NCOF=(NTAP-1)/2+1
        ELSE
          ICASE=2
          NCOF=NTAP/2
        END IF
      ELSE
        IF (MOD(NTAP,2).NE.0) THEN
          ICASE=3
          NCOF=(NTAP-1)/2
        ELSE
          ICASE=4
          NCOF=NTAP/2
        END IF
      END IF

* Convert the band specifications into a specification on a
* frequency grid
      IF (GRIDD.EQ.-1) THEN
        NGRIDD=NGMAX
      ELSE IF (GRIDD.LE.0.0) THEN
        NGRIDD=NCOF*GRIDD0
      ELSE
        NGRIDD=NCOF*GRIDD
      END IF
      CALL GRIDPT(ICASE,NGRIDD,IBAND,NBANDS,
     -            FREQ,VAL,WEIGHT,VLIML,VLIMU,
     -            NGMAX,NGRID,FGRID,DES,WT,DLIML,DLIMU)


*********************

* Modify the specifications
* - Receiving filters - sin(x)/x compensation
* - Differentiators - change slope specification to value
* - Others - no change
* (DES, WT, DLIML, and DLIMU are transformed in place)
      CALL CHSPEC(ITYPE,NGRID,FGRID,DES,DES,WT,WT,
     -            DLIML,DLIML,DLIMU,DLIMU)

*********************

* There are four cases to be considered for the linear phase filter
*         N-1       -n
*  H(z) = SUM h(n) z
*         n=0
* Case     symmetry      even/odd
*  1)    h(n)=h(N-1-n)    N odd         bandpass filter
*  2)    h(n)=h(N-1-n)    N even        bandpass filter
*  3)    h(n)=-h(N-1-n)   N odd         differentiator/Hilbert tr.
*  4)    h(n)=-h(N-1-n)   N even        differentiator/Hilbert tr.
* The frequency response, H(w) = H(exp(jw)), can be written in the
* following form for each of the cases,
*                            M-1
* H(w) = exp(jw(N-1)/2) Q(w) SUM a(n) cos(wn) .
*                            n=0
*
* Case     Q(w)       No. coef. (M)
*  1)        1          (N-1)/2+1
*  2)    2 cos(w/2)      N/2
*  3)    2j sin(w)      (N-3)/2+1
*  4)    2j sin(w/2)     N/2
* The given specifications will be converted to an equivalent
* specification for approximation by a sum of cosines.
* The phase factors in H(w) do not enter into the magnitude
* approximation.  However, the fixed factor Q(w) must be absorbed into
* the desired value, constraint values, and weighting.

* Transform the specifications for the different filter types, taking
* the factor Q(w) into account
* (DES, WT, DLIML, and DLIMU are transformed in place)
      CALL TRSPEC(ICASE,NGRID,FGRID,XGRID,DES,DES,WT,WT,
     -            DLIML,DLIML,DLIMU,DLIMU)


      RETURN

      END
