*------------- Telecommunications & Signal Processing Lab --------------
*                          McGill University
*
*
* Module:
*     SUBROUTINE BANDPT (FREQ, VAL, WEIGHT, VLIML, VLIMU, NVAL,
*                        FMIN, FMAX, DELF, MAXNPT,
*                        NPT, FGRID, DES, WT, DLIML, DLIMU)
*
*
* Purpose:
*     Determine the specifications for frequency bands
*
*
* Description:
*     Given the input band specification, this routine generates an
*     equivalent specification on an array of frequency points.  The
*     intermediate values are generated by interpolating using a
*     piecewise monotonic cubic function between the values of the
*     input specifications.
*
*
* Parameters:
* R ->  FREQ   - Array of normalized frequencies.  These values must be
*                in increasing order.
* R ->  VAL    - Array of desired values
* R ->  WEIGHT - Array of weights
* R ->  VLIML  - Array of lower constraints
* R ->  VLIMU  - Array of upper constraints
* I ->  NVAL   - Number of values in each of FREQ, VAL, WEIGHT, VLIML,
*                and VLIMU
* R ->  FMIN   - Minimum frequency allowed
* R ->  FMAX   - Maximum frequency allowed
* R ->  DELF   - Desired frequency increment
* I ->  MAXNPT - Largest permissible value for NPT
* I <-  NPT    - Number of data points generated
* R <-  FGRID  - Array specifying the normalized frequency values
* R <-  DES    - Array of desired values on the frequency grid
* R <-  WT     - Array of desired weights on the frequency grid
* R <-  DLIML  - Array of lower constraints
* R <-  DLIMU  - Array of upper constraints
*
*
* Routines required:
*     HALT   - Print an error message, stop with error status set
*     ICEILX - Calculate the ceiling function
*     PMCINT - Generate a piecewise monotonic cubic interpolant
*
*
* Author / revision:
*     P. Kabal  Copyright (C) 1993
*     $Revision: 1.3 $  $Date: 1993/01/25 05:05:08 $
*
*
*-----------------------------------------------------------------------

      SUBROUTINE BANDPT (FREQ, VAL, WEIGHT, VLIML, VLIMU, NVAL,
     -                     FMIN, FMAX, DELF, MAXNPT,
     -                     NPT, FGRID, DES, WT, DLIML, DLIMU)


      INTEGER NVAL,MAXNPT,NPT
      INTEGER N,L,NS,NP,K
      INTEGER ICEILX

      REAL FREQ(NVAL),VAL(NVAL),WEIGHT(NVAL),VLIML(NVAL),VLIMU(NVAL),
     -       FMIN,FMAX,DELF,FGRID(MAXNPT),DES(MAXNPT),WT(MAXNPT),
     -       DLIML(MAXNPT),DLIMU(MAXNPT)
      REAL FL,FU

      DOUBLE PRECISION A


* Band specification: single point
      IF (NVAL.EQ.1) THEN
        FGRID(1)=MIN(MAX(FREQ(1),FMIN),FMAX)
        IF (MAXNPT.LT.1)
     -    CALL HALT('BANDPT - Insufficient array space')
        N=1

* Multiple point band specification
      ELSE
        N=0
        DO 260 L=1,NVAL-1

* Set the frequency limits for the interval
          IF (FREQ(L+1).LT.FREQ(L))
     -      CALL HALT('BANDPT - Frequency values not in '//
     -                'increasing order')
          FL=MIN(MAX(FREQ(L),FMIN),FMAX)
          FU=MIN(MAX(FREQ(L+1),FMIN),FMAX)

* Calculate NS, the number of samples for the interval [FL,FU)
* NP is the number of samples actually used.  It is equal
* to NS for all but the last interval in a band.  In that
* interval, one extra point at FU is generated.
* The frequency grid is uniformly spaced between FL and FU
          NS=ICEILX((FU-FL)/DELF)
          IF (NS.EQ.0)
     -      CALL HALT('BANDPT - Frequencies too closely spaced')
          IF (L.EQ.NVAL-1) THEN
            NP=NS+1
          ELSE
            NP=NS
          END IF
          IF (N+NP.GT.MAXNPT)
     -      CALL HALT('BANDPT - Insufficient array space '//
     -                'for the given grid density')
          DO 240 K=0,NP-1
            N=N+1
            A=DBLE(K)/NS
            FGRID(N)=A*FU + (1D0-A)*FL
 240      CONTINUE

 260    CONTINUE

      END IF

* Linear interpolation of the values and weights
      CALL PMCINT(FREQ,VAL,NVAL,FGRID,DES,N)
      CALL PMCINT(FREQ,WEIGHT,NVAL,FGRID,WT,N)
      CALL PMCINT(FREQ,VLIML,NVAL,FGRID,DLIML,N)
      CALL PMCINT(FREQ,VLIMU,NVAL,FGRID,DLIMU,N)

      NPT=N


      RETURN

      END
