/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void INwriteCof (const char Fname[], const float h[], int Ncof, double Delay,
                   int Ir, const struct PSD *psd, const struct PSD *dis)

Purpose:
  Write an interpolating filter coefficient file

Description:
  This routine writes a coefficient file for an interpolating filter.  The data
  written to the file includes a header containing the design specification.
  This header is followed by the filter coefficients.  The file name is also
  printed on standard output.

Parameters:
   -> const char Fname[]
      File name for the coefficient file
   -> const float h[]
      Array of coefficients (Ncof values)
   -> int Ncof
      Number of filter coefficients
   -> double Delay
      Filter delay (samples)
   -> int Ir
      Interpolating ratio
   -> const struct PSD *psd
      Structure describing the continuous power spectral density.  This
      structure contains an array of normalized frequencies and an array of
      the corresponding power spectral values.
   -> const struct PSD *dis
      Structure describing the discrete (sinusoidal) components of the power
      spectrum.  This structure contains an array of normalized frequencies and
      an array of the corresponding powers of the sinusoids.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/03/01 14:45:48 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INwriteCof.c 1.10 1996/03/01 FilterDesign-V1R7a $";

#include <stdio.h>
#include <libtsp.h>
#include "DFiltInt.h"

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

void
INwriteCof (Fname, h, Ncof, Delay, Ir, psd, dis)

     const char Fname[];
     const float h[];
     int Ncof;
     double Delay;
     int Ir;
     const struct PSD *psd;
     const struct PSD *dis;

{
  FILE *fp;
  char Fullname[FILENAME_MAX+1];
  int i;

/* Open the coefficient file, print the name to stdout */
  FLbackup (Fname);
  fp = fopen (Fname, "w");
  if (fp == NULL)
    UTerror ("%s: Unable to open coefficient file", PROGRAM);
  FLfullName (Fname, Fullname);
  printf ("Coefficient file: %s\n", Fullname);

/* Preamble */
  fprintf (fp, "!FIR - Interpolating filter\n");
  fprintf (fp, "! Interpolating Ratio: %d, Delay: %.3g\n", Ir, Delay);
  fprintf (fp, "! Power Spectrum:\n");

/* Design specifications */
  fprintf (fp, "!     Freq.      Value");
  for (i = 1; i < MINV (3, MAXV (psd->N, dis->N)); ++i)
    fprintf (fp, "         Freq.      Value");
  fprintf (fp, "\n");

  for (i = 0; i < psd->N; ++i) {
    if (i % 3 == 0)
      fprintf (fp, "!");
    else
      fprintf (fp, "    ");
    fprintf (fp, "%10.5f%11.5g", psd->f[i], psd->psd[i]);
    if ((i + 1) % 3 == 0 || i + 1 == psd->N)
      fprintf (fp, "\n");
  }

  for (i = 0; i < dis->N; ++i) {
    if (i % 3 == 0)
      fprintf (fp, "!");
    fprintf (fp, "%10.5f%11.5g cos", dis->f[i], dis->psd[i]);
    if ((i + 1) % 3 == 0 || i + 1 == dis->N)
      fprintf (fp, "\n");
  }

/* Coefficient values */
  for (i = 0; i < Ncof; ++i) {
    fprintf (fp, "%15.7g", h[i]);
    if ((i + 1) % 5 == 0 || i + 1 == Ncof)
      fprintf (fp, "\n");
  }

  fclose (fp);
}
