/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void INprintSpec (int Ncof, double Delay, int Ir, const float mse[],
                    double Sfreq, const struct PSD *psd, const struct PSD *dis)

Purpose:
  Print design parameters for an interpolating filter

Description:
  This routine prints the design parameters for a minimum mean-square error
  interpolation filter.

Parameters:
   -> int Ncof
      Number of filter coefficients
   -> double Delay
      Filter delay (samples)
   -> int Ir
      Interpolating ratio
   -> double Sfreq
      Sampling frequency
   -> const float mse[]
      Array of Ir normalized mean-square interpolation errors
   -> const struct PSD *psd
      Structure describing the continuous power spectral density.  This
      structure contains an array of normalized frequencies and an array of
      the corresponding power spectral values.
   -> const struct PSD *dis
      Structure describing the discrete (sinusoidal) components of the power
      spectrum.  This structure contains an array of normalized frequencies and
      an array of the corresponding powers of the sinusoids.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.6 $  $Date: 1996/06/17 21:29:47 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: INprintSpec.c 1.6 1996/06/17 FilterDesign-V1R7a $";

#include <math.h>
#include <stdio.h>
#include "DFiltInt.h"

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))
#define DBP(x)		(10.0 * log10 (x))

void
INprintSpec (Ncof, Delay, Ir, mse, Sfreq, psd, dis)

     int Ncof;
     double Delay;
     int Ir;
     const float mse[];
     double Sfreq;
     const struct PSD *psd;
     const struct PSD *dis;

{
  int i;
  double amse;


/* Preamble */
  printf ("FIR - Interpolating filter: %d coefficients\n", Ncof);
  printf (" Interpolating Ratio: %d, Delay: %.3g\n", Ir, Delay);
  printf (" Sampling Frequency: %.4g\n", Sfreq);
  printf (" Power Spectrum:\n");

/* Design specifications */
  printf ("     Freq.      Value");
  for (i = 1; i < MINV (3, MAXV (psd->N, dis->N)); ++i)
    printf ("         Freq.      Value");
  printf ("\n");

  for (i = 0; i < psd->N; ++i) {
    if (i % 3 != 0)
      printf ("    ");
    printf ("%10.5g%11.5g", Sfreq * psd->f[i], psd->psd[i]);
    if ((i + 1) % 3 == 0 || i + 1 == psd->N)
      printf ("\n");
  }

  for (i = 0; i < dis->N; ++i) {
    printf ("%10.5g%11.5g cos", Sfreq * dis->f[i], dis->psd[i]);
    if ((i + 1) % 3 == 0 || i + 1 == dis->N)
      printf ("\n");
  }

/* Print the mean-square error */
  printf (" Normalized mean-square error:\n");
  amse = 0.0;
  for (i = 0; i < Ir; ++i) {
    printf ("   Subfilter %d: %.3g\n", i,  mse[i]);
    amse = amse + mse[i];
  }
  amse = amse / Ir;

  printf (" Average normalized mean-square error: %.3g", amse);
  if (amse != 0)
    printf (" (SNR = %.4g dB)\n", -DBP (amse));
  else
    printf ("\n");
}
