/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Routine:
  void INoptions (int argc, const char *argv[], int *Ncof, float *Delay,
                  int Ir, float *Sfreq, const char **Fname)

Purpose:
  Decode options for DFiltInt

Description:
  This routine decodes options for DFiltInt.

Parameters:
   -> int argc
      Number of command line arguments
   -> const char *argv[]
      Array of pointers to argument strings
  <-  int *Ncof
      Number of filter coefficients
  <-  float *Delay
      Filter delay (samples)
  <-  int *Ir
      Interpolation ratio
  <-  float *Sfreq
      Sampling frequency of the input sequence
  <-  const char **Fname
      Coefficient file name

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.15 $  $Date: 1996/03/01 14:44:59 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: INoptions.c 1.15 1996/03/01 FilterDesign-V1R7a $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include "DFiltInt.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define ERRSTOP(text,par)	UThalt ("%s: %s: \"%s\"", PROGRAM, text, par)

/* Option tables and usage message */
#define LOPT	((sizeof OptTable) / (2 * (sizeof OptTable[0])))
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-i#", "--i*nterpolate=",
  "-n#", "--n*umber_coefficients=",
  "-d#", "--d*elay=",
  "-s#", "--s*rate=",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] [FiltFile]\n\
Options:\n\
  -i IR, --interpolate=IR     Interpolation ratio.\n\
  -n NCOF, --number_coefficients=NCOF\n\
                              Number of filter coefficients.\n\
  -d DELAY, --delay=DELAY     Filter delay.\n\
  -s SFREQ, --srate=SFREQ     Sampling frequency (of the input sequence).\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
INoptions (argc, argv, Ncof, Delay, Ir, Sfreq, Fname)

     int argc;
     const char *argv[];
     int *Ncof;
     float *Delay;
     int *Ir;
     float *Sfreq;
     const char **Fname;

{
  int Index;
  const char *OptArg;
  const char **optt;

  int nF;
  int ir;
  int ncof;
  int delayset;
  float delay;
  float sfreq;
  int n, nn;

/* Defaults */
  delayset = 0;
  sfreq = 1;
  *Fname = NULL;

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;
  ir = 0;
  ncof = 0;

/* Decode options */
  Index = 1;
  optt = OptTable;
  while (Index < argc) {
    n = UTgetOption (&Index, argc, argv, optt, &OptArg);
    nn = ((n + 3) / 2) - 1;	/* n = -2 ==> nn = -1 */
    switch (nn) {
    case 0:
      /* Coefficient file */
      ++nF;
      if (nF > 1)
        UThalt ("%s: Too many filenames specified", PROGRAM);
      *Fname = OptArg;
      break;
    case 1:
      /* Interpolation factor */
      if (STdec1int (OptArg, &ir) || ir <= 0)
	ERRSTOP ("Invalid interpolation factor", OptArg);
      break;
    case 2:
      /* Number of coefficents */
      if (STdec1int (OptArg, &ncof) || ncof <= 0)
	UThalt ("%s: Invalid number of coeffients", PROGRAM);
      break;
    case 3:
      /* Filter delay */
      if (STdec1float (OptArg, &delay))
	UThalt ("%s: Invalid delay value", PROGRAM);
      delayset = 1;
      break;
    case 4:
      /* Sampling frequency */
      if (STdec1float (OptArg, &sfreq) || sfreq <= 0.0)
	ERRSTOP ("Invalid sampling frequency", OptArg);
      break;
    case LOPT-2:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case LOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case LOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UThalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (ir <= 0) {
    UTwarn ("%s - Interpolating ratio not specified", PROGRAM);
    UThalt (Usage, PROGRAM);
  }
  if (ncof <= 0) {
    UTwarn ("%s - Number of coefficients not specified", PROGRAM);
    UThalt (Usage, PROGRAM);
  }
  if (! delayset)
    delay = 0.5 * (ncof - 1);

/* Set return values */
  *Ncof = ncof;
  *Ir = ir;
  *Delay = delay;
  *Sfreq = sfreq;
}
