/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void MMprintSpec (int Ftype, double dev, struct Bspec *S)

Purpose:
  Print FIR filter design parameters

Description:
  This routine prints the design parameters for an FIR filter.

Parameters:
   -> int Ftype
      Filter type
   -> double dev
      Weighted filter deviation
   -> struct Bspec *S
      Band specifications

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.4 $  $Date: 1995/11/24 01:40:44 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMprintSpec.c 1.4 1995/11/24 FilterDesign-V1R7a $";

#include <math.h>
#include <stdio.h>
#include <libtsp.h>
#include "DFiltFIR.h"

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))
#define DBV(x)		(20.0 * log10 (x))

void
MMprintSpec (Ftype, dev, S)

     int Ftype;
     double dev;
     const struct Bspec *S;

{
  double devx, devdB;
  int i, j, k;

/* Preamble */
  if (Ftype == BPF  || Ftype == REC) {
    if (Ftype == BPF)
      printf ("Linear Phase FIR Filter\n");
    else
      printf ("Linear Phase FIR Filter (sin(x)/x compensation)\n");
    printf ("           Freq.      Value    Weight           ");
    printf ("Limits      Deviation  Dev dB\n");
  }
  else if (Ftype == DIF) {
    printf ("Differentiator\n");
    printf ("           Freq.      Slope    Weight           ");
    printf ("Limits      Deviation\n");
  }
  else if (Ftype == HIL) {
    printf ("Hilbert Transform Filter\n");
    printf ("           Freq.      Value    Weight           ");
    printf ("Limits      Deviation\n");
  }
  else
    UThalt ("MMprintSpec: Invalid filter type");

/* Design specifications */
  i = 0;
  for (k = 1; k <= S->Nbands; ++k) {
    for (j = 0; j < S->Nval[k-1]; ++j, ++i) {
      if (j == 0)
        printf ("Band %2d: ", k);
      else
        printf ("         ");

      /* Freq / Value / Weight */
      printf ("%7.4f %10.3g %9.2g ", S->f[i], S->val[i], S->w[i]);

      /* Limits */
      if (S->liml[i] != NOLIMIT_L)
	printf ("%10.3g ", S->liml[i]);
      else
	printf ("     ----  ");
      if (S->limu[i] != NOLIMIT_U)
        printf ("%10.3g ", S->limu[i]);
      else
        printf ("     ----  ");

      /* Calculate the deviation */
      devx = dev / S->w[i];
      devx = MAXV (MINV (devx, S->val[i] - S->liml[i]),
                   MINV (devx, S->limu[i] - S->val[i]));
      printf ("%9.2g", devx);
 
      if (Ftype == BPF || Ftype == REC)
	if (devx > 0) {
	  if (S->val[i] != 0.0)
	    devdB = DBV ((double) (devx + S->val[i]) / S->val[i]);
	  else
	    devdB = DBV ((double) devx);
	  printf (" %7.2f\n", devdB);
	}
	else
	  printf ("    ****\n");
      else
        printf ("\n");

    }
  }
}
