/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int MMexchExt (const struct Pcev *P, const struct Gspec *G, double dev,
                 int Ext[], int Etype[])

Purpose:
  Find extrema of the error

Description:
  The routine finds the locations of the extrema of the error function for a
  constrained weighted minimax polynomial fit to a tabulated function.  This
  routine is intended to be used as part of a Remez exchange algorithm.  The
  desired values are specified on a dense grid of points.  Given an initial
  guess at the locations of a set of extremal points with alternating errors,
  this routine changes the locations of the extrema to coincide with the local
  maxima on the grid.

Parameters:
  <-  int MMexchExt
      Number of extrema changed.  A zero value indicates that the Remez
      exchange algorithm has converged.
   -> const struct Pcev *P
      Structure containing the array of Lagrange interpolation coefficients
      and the corresponding abscissa and ordinate values.  These values
      together specify the polynomial that passes through the points specified.
   -> const struct Gspec *G.
      Structure with the arrays specifying the specifications on a dense grid
      of points
   -> double dev
      Deviation at the design points.  This is the value of the deviation as
      determined by MMmmDev.
  <-> int Ext[]
      Indices of the extremal points (P->N values).  Each index is a value
      from 0 to G->Ngrid-1.  These values are both input and output. The given
      input values are used as starting points and then updated by this
      routine.
  <-> int Etype[]
      Type of extremum (P->N values) for each of the extremal points in array
      Ext.  These values are updated by this routine.
      -2 - Lower constraint limit
      -1 - Lower ripple
      +1 - Upper ripple
      +2 - Upper constraint limit

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.7 $  $Date: 1995/05/26 00:21:39 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMexchExt.c 1.7 1995/05/26 FilterDesign-V1R7a $";

#include "DFiltFIR.h"

/* Internal routine prototypes */
static int
MM_errMax p_((int kst, int kfn, int step, float *err, double dev, int Etype[],
	      const struct Pcev *P, const struct Gspec *G));
static double
MM_wgtErr p_((double y, double dev, int *Etype, float des, float wt,
	      float liml, float limu));

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

/* step codes */
#define SUP	1	/* Step up */
#define CUP	2	/* Search upward */
#define SDOWN	-1	/* Step up */
#define CDOWN	-2	/* Search downward */

#define EPS	1e-5

int
MMexchExt (P, G, dev, Ext, Etype)

     const struct Pcev *P;
     const struct Gspec *G;
     double dev;
     int Ext[];
     int Etype[];

{
  int i;
  int step;
  float err, err0, errN, err0T;
  int EtypeB, EtypeT;
  int Ngrid;
  int Next;
  int Nchange;
  int k, kst, kfn;
  int klow, kup;
  int kext, kext0, kextN, kextB, kextT;

  Next = P->N;
  Ngrid = G->Ngrid;
  Nchange = 0;
  kext0 = Ext[0];
  kextN = Ext[Next-1];

/* *************** Search for the Next largest alternating extrema */
  klow = -1;
  for (i = 0; i < Next; ++i) {

    if (i == Next - 1)
      kup = Ngrid;
    else
      kup = Ext[i+1];
    k = Ext[i];

    err = MM_wgtErr (P->y[i], dev, &Etype[i], G->des[k], G->wt[k],
		     G->liml[k], G->limu[k]);

    do {		/* one-trip loop to allow breaks */

/* Look for a local maximum by increasing the grid index */
      kst = k + 1;
      kfn = kup - 1;
      step = SUP;
      kext = MM_errMax (kst, kfn, step, &err, dev, &Etype[i], P, G);
      if (kext >= 0)
        break;

/* Look for a local maximum by decreasing the grid index */
      kst = k - 1;
      kfn = klow + 1;
      step = SDOWN;
      kext = MM_errMax (kst, kfn, step, &err, dev, &Etype[i], P, G);
      if (kext >= 0)
	break;

/* Keep looking for a local maximum, decreasing the grid index */
      if (Nchange == 0) {
	kst = k - 2;
	kfn = klow + 1;
	step = CDOWN;
	kext = MM_errMax (kst, kfn, step, &err, dev, &Etype[i], P, G);
	if (kext >= 0)
	  break;
      
/* Keep looking for a local maximum, increasing the grid index */
	kst = k + 2;
	kfn = kup - 1;
	step = CUP;
	kext = MM_errMax (kst, kfn, step, &err, dev, &Etype[i], P, G);
	if (kext >= 0)
	  break;
      }

    } while (0);	/* one-trip only */

/* Update the extremum position */
    if (kext >= 0) {
      klow = MAXV (kext, k);
      Ext[i] = kext;
      ++Nchange;
    }
    else
      klow = k;

/* Save the error for the first extremum */
    if (i == 0)
      err0 = err;
  }

/* Save the error for the last extremum */
  errN = err;

/* *************** Endpoints search */
/*
   Search below the lower endpoint for a local maximum with an error larger
   than the error found at the upper endpoint
*/
  err = errN * (1.0 + EPS);
  kst = 0;
  kfn = MINV (kext0, Ext[0]) - 1;
  step = CUP;
  if (Etype[0] == LDEV || Etype[0] == LLIM)
    EtypeB = UDEV;
  else
    EtypeB = LDEV;
  kextB = MM_errMax (kst, kfn, step, &err, dev, &EtypeB, P, G);
  if (kextB >= 0)
    err0T = MAXV (err0, err);
  else
    err0T = err0;

/*
   Search above the upper endpoint for a local maximum with an error larger
   than the error found at or below the lower endpoint
*/
  err = err0T * (1.0 + EPS);
  kst = Ngrid - 1;
  kfn = MAXV (kextN, Ext[Next-1]) + 1;
  step = CDOWN;
  if (Etype[Next-1] == LDEV || Etype[Next-1] == LLIM)
    EtypeT = UDEV;
  else
    EtypeT = LDEV;
  kextT = MM_errMax (kst, kfn, step, &err, dev, &EtypeT, P, G);

  if (kextT >= 0) {
    /* Found a local maximum above the former upper endpoint */
    for (i = 0; i < Next - 1; ++i) {
      Ext[i] = Ext[i+1];
      Etype[i] = Etype[i+1];
    }
    Ext[Next-1] = kextT;
    Etype[Next-1] = EtypeT;
    ++Nchange;
  }
  else if (kextB >= 0) {
    /* Found a local maximum below the former lower endpoint */
    for (i = Next - 2; i >= 0; --i) {
      Ext[i+1] = Ext[i];
      Etype[i+1] = Etype[i];
    }
    Ext[0] = kextB;
    Etype[0] = EtypeB;
    ++Nchange;
  }

  return Nchange;
}

/*
Routine:
  static int MM_errMax (int kst, int kfn, int step, float *err, double dev,
                        int Etype[], const struct Pcev *P,
                        const struct Gspec *G)

Description:
  This routine looks for a local maximum in the error curve.  The routine
  checks a range of values on the approximating grid.

Parameters:
  <-  int MM_errMax
      Position of the extremum found (in the range kst to kfn).  This value is
      set to -1 if no extremum with error larger than the initial value of
      err is found.
   -> int kst
      Beginning position (0 to Ngrid-1)
   -> int kfn
      Search limit (0 to Ngrid-1).  The search starts at kst in the direction
      indicated by step.  If kst is on the wrong side of kfn (for instance
      kfn < kst when step is positive), no points are searched.
   -> int step
      Search mode.  The sign of is indicates the search direction, while the
      magnitude of is indicates the type of search.
      step > 0 - upwards from kst
      step < 0 - downwards from kst
      |step|=1 - Continue searching as long as the error is increasing.  If
                 the first probe is not successful, the search is terminated.
      |step|=2 - Continue searching until a local maximum is found or the
                 search range is exhausted
  <-> float *err
      This is the largest error found.  It is both an input and an output
      variable.
   -> double dev
      Deviation at the design points
  <-> int Etype[]
      Type of extremum.  This is both an input and an output value.  On input
      it indicates the type of extremum to which err applies.  On output it
      indicates the type of extremum found.
      -2 - Lower constraint limit
      -1 - Lower ripple
      +1 - Upper ripple
      +2 - Upper constraint limit
   -> const struct Pcev *P
      Structure containing the array of Lagrange interpolation coefficients
      and the corresponding abscissa and ordinate values.  These values
      together specify the polynomial that passes through the points specified.
   -> const struct Gspec *G.
      Structure with the arrays specifying the specifications on a dense grid
      of points
*/

static int
MM_errMax (kst, kfn, step, err, dev, Etype, P, G)

     int kst;
     int kfn;
     int step;
     float *err;
     double dev;
     int Etype[];
     const struct Pcev *P;
     const struct Gspec *G;

{
  int cont;
  float errt;
  int kext, k, id, ll;
  double px;

  kext = -1;
  if (step == SUP || step == CUP)
    id = 1;
  else
    id = -1;
  cont = 1;
  if (step == SUP || step == SDOWN)
    cont = 0;

  for (k = kst; id < 0 ? k >= kfn : k <= kfn; k += id) {

    px = MMintVal ((double) G->grid[k], P);

/* Calculate the weighted error */
    ll = *Etype;
    errt = MM_wgtErr (px, dev, &ll, G->des[k], G->wt[k], G->liml[k],
		      G->limu[k]);

/* See if the error has increased */
    if (errt > *err) {
      kext = k;
      *Etype = ll;
      *err = errt;
    }
    else if (kext >= 0 || cont == 0)
      break;
  }

  return kext;
}

/*
Routine:
  static double MM_wgtErr (double y, double dev, int *Etype, float des,
                           float wt, float liml, float limu)

Description:
  This routine calculates the weighted error for a given ordinate value.  The
  errors for values which correspond to values which should lie above the
  desired value are calculated as
    Ee = wt*(y-des)     - ordinary value
    El = (y-limu)+dev   - value violating the constraints
  Similar expressions are applied for values which should lie below the desired
  value.  The first formula is used, except if the value y violates the
  constraints.  As the weighted error converges to become minimax, the ordinary
  extremal points have weighted deviation equal to dev.  With this error
  expression, the extrema which approach the constraints also give errors which
  approach the value dev.

  The input parameter Etype determines whether the value should lie above or
  below the desired value.  Both El and Eo are calculated.  If El > Eo, the
  value of El is used.  The value of Etype is set according to which of El or
  Eo is used.
*/

static double
#ifdef __STDC__
MM_wgtErr (double y, double dev, int *Etype, float des, float wt, float liml,
	   float limu)
#else
MM_wgtErr (y, dev, Etype, des, wt, liml, limu)
     double y;
     double dev;
     int *Etype;
     float des;
     float wt;
     float liml;
     float limu;
#endif

{
  float errl, err;
  int Et;

  Et = *Etype;
  if (Et == UDEV || Et == ULIM) {

    /* Looking for an upward peak */
    err = wt * (y - des);
    errl = (y - limu) + dev;
    Et = UDEV;
    if (errl > err) {
      Et = ULIM;
      err = errl;
    }
  }
  else {

    /* Looking for a downward peak */
    err = wt * (des - y);
    errl = (liml - y) + dev;
    Et = LDEV;
    if (errl > err) {
      Et = LLIM;
      err = errl;
    }
  }

  *Etype = Et;
  return err;
}
