/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int MMdesFIR (int Ftype, double Gridd, const struct Bspec *S, float h[],
                int Ncof, float *devs)

Purpose:
  Design a minimax linear phase FIR filter

Description:
  This routine designs multiple band bandpass filters, differentiators, or
  Hilbert transform filters using the Remez exchange algorithm.  This program
  implements the filter design procedure outlined by McClellan, Parks and
  Rabiner.  This procedure has been modified to include constraints as
  described by Grenez.
  References:
    J. H. McClellan, T.W. Parks and L.R. Rabiner,
    "A computer program for designing optimum FIR linear phase digital
    filters", IEEE Trans. Audio and Electroacoustics,
    vol. AU-21, pp. 506-526, December 1973.
    F. Grenez,
    "Design of linear or minimum-phase FIR filters by constrained Chebyshev
    approximation", Signal Processing,
    vol. 5, pp. 325-332, July 1983.

Parameters:
  <-  int MMdesFIR
      Error parameter coded as follows.
       0  - No error
       1  - Remez algorithm failed to converge, the deviation is not
            monotonically increasing
       2  - Remez algorithm failed to converge, too many iterations
      For these cases, this routine returns coefficient values.  For all other
      errors, an error message is printed and execution is halted.
   -> int Ftype
      Filter type.
      1 - Multiple passband/stopband filter
      2 - Multiple passband/stopband filter (sin(x)/x compensation)
      3 - Differentiator
      4 - Hilbert transform filter
   -> double Gridd
      Grid density.  The number of points in the frequency grid used to
      determine the positions of the extrema of the approximating function is
      the product of Gridd and the number of extrema (approximately Ncof/2).
   -> struct Bspec *S
      Band specifications
  <-  float h[]
      Array of resultant filter coefficients
   -> int Ncof
      Number of filter coefficients desired
  <-  float *devs
      Resultant deviation from the desired specifications.  The actual
      deviation at a given frequency is obtained by dividing devs by the
      weight at that frequency.

Author / revision:
  P. Kabal  Copyright (C) 1995
  $Revision: 1.3 $  $Date: 1995/02/09 22:07:56 $

/
-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MMdesFIR.c 1.3 1995/02/09 FilterDesign-V1R7a $";

#include <libtsp.h>
#include "DFiltFIR.h"

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))

int
MMdesFIR (Ftype, Gridd, S, h, Ncof, devs)

     int Ftype;
     double Gridd;
     const struct Bspec *S;
     float h[];
     int Ncof;
     float *devs;

{
  int i;
  int Fcase;
  int Ngrid, Next, Npar;
  int ngridd;
  int *Etype;
  double *alpha, *x, *y;
  double *hd;
  int ier;
  struct Gspec *G;

/* Resolve the filter type */
  if (Ftype == BPF || Ftype == REC) {
    if (Ncof % 2 != 0) {
      Fcase = 1;
      Npar = (Ncof + 1) / 2;
    }
    else {
      Fcase = 2;
      Npar = Ncof / 2;
    }
  }
  else if (Ftype == DIF || Ftype == HIL) {
    if (Ncof % 2 != 0) {
      Fcase = 3;
      Npar = (Ncof - 1) / 2;
    }
    else {
      Fcase = 4;
      Npar = Ncof / 2;
    }
  }
  else
    UThalt ("MMdesFIR: Invalid filter type");

/* Convert to a specification on a dense grid of frequencies */
  ngridd = Npar * Gridd;
  G = MMgridVal (Fcase, ngridd, S);

/*
  Modify the specifications
  - Receiving filters - sin(x)/x compensation
  - Differentiators - change slope specification to value
*/
  MMmodSpec (Ftype, G, G);

/*
   Transform the specifications for the different cases (odd/even number of
   coefficients, odd/even symmetry) into a common form for approximation by
   a sum of cosines 
*/
  MMcanonSpec (Fcase, G, G);

  Ngrid = G->Ngrid;
  Next = MINV (Ngrid, Npar + 1);

/* Allocate work arrays */
  x = (double *) UTmalloc (2 * Next * sizeof (double));
  y = x + Next;
  Etype = (int *) UTmalloc (Next * sizeof (int));
  alpha = (double *) UTmalloc (Npar * sizeof (double));
  hd = (double *) UTmalloc (Ncof * sizeof (double));

/* ********** Filter design */
/* Find the extremal values for the minimax approximation */
  ier = MMdesRemez (G, x, y, Etype, Next, devs);

/* ********** */

/* Find the coefficients of the approximation */
  MMcosCof ((double) G->grid[0], (double) G->grid[Ngrid-1], x, y, Next, alpha);
  for (i = Next - 1; i < Npar; ++i)
    alpha[i] = 0.0;

/* Calculate the impulse response */
  MMfiltCof (Fcase, alpha, hd, Ncof);

/* Copy the impulse response to the output array */
  for (i = 0; i < Ncof; ++i)
    h[i] = hd[i];

/* Deallocate the arrays */
  UTfree (hd);
  UTfree (alpha);
  UTfree (Etype);
  UTfree (x);
  UTfree (G->grid);
  UTfree (G);

  return ier;
}
