/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void AFupdAIhead (const AFILE *AFp)

Purpose:
  Update header information in an AIFF-C audio file

Description:
  This routine updates the data length fields of an AIFF-C audio file.  The
  file is assumed to have been opened with routine AFopenWrite.

Parameters:
   -> const AFILE *AFp
      Audio file pointer for an audio file opened by AFopenWrite

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.5 $  $Date: 1996/08/14 18:14:49 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFupdAIhead.c 1.5 1996/08/14 AFsp-V2R2 $";

#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>
#include <libtsp/WVpar.h>

#define WHEAD_V(fp,offs,value,swap) \
			AFwriteHead (fp, (long int) (offs), \
				     (void *) &(value), \
				     sizeof (value), 1, swap)

static const int Lenb[NFD] = {
  0,
  FDL_MULAW8,
  FDL_ALAW8,
  FDL_UINT8,
  FDL_INT8,
  FDL_INT16,
  FDL_FLOAT32,
  FDL_TEXT
};

void
AFupdAIhead (AFp)

     const AFILE *AFp;

{
  long int Nbytes;
  uint4_t Size, numSampleFrames;
  static const uint1_t Pad = 0;

/* Checks */
  if (AFp->Op != FO_WO)
    UThalt ("AFupdAIhead: File not opened for write");
  if (AFp->Ftype != FT_AIFF_C)
    UThalt ("AFupdAIhead: Not an AIFF-C file");

/* Add a padding byte to the sound data; this padding byte is not included
   in the SSND chunk ckDataSize field, but is included in the FORM chunk
   ckDataSize field
*/
  Nbytes = AFp->End;
  if (Nbytes % 2 != 0)
    Nbytes += WHEAD_V (AFp->fp, Nbytes, Pad, DS_EB);

/* Update the FORM chunk ckDataSize field */
  Size = (uint4_t) (Nbytes - 8);
  WHEAD_V (AFp->fp, 4L, Size, DS_EB);			/* File length - 8 */

/* Update the COMM chunk numSampleFrames field:
   The numSampleFrames field is at offset 22 if the COMM chunk is the first
   chunk in the FORM chunk
*/
  if (AFp->Format <= 0 || AFp->Format >= NFD)
    UThalt ("AFupdAIhead: Invalid data format code");
  numSampleFrames = (uint4_t) ((AFp->End - AFp->Start) /
                               (Lenb[AFp->Format] * AFp->Nchan));
  WHEAD_V (AFp->fp, 22L, numSampleFrames, DS_EB);

/* Update the SSND chunk ckDataSize field:
   The ckDataSize is 12 bytes before the start of data if SSND.offset = 0
 */
  Size = (uint4_t) (AFp->End - AFp->Start + 8);
  WHEAD_V (AFp->fp, AFp->Start - 12, Size, DS_EB);	/* Data length */

  return;
}
