/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFsetNHpar (const char Fname[], int Format, long int Nchan,
                     double Sfreq, FILE *fpout)

Purpose:
  Set file format information for a headerless audio file

Description:
  This routine sets the file format information for a headerless audio file.
  The file is assumed to have been opened with routine AFopenWrite.
  Optionally, the file format information is printed.

Parameters:
  <-  AFILE AFsetNHpar
      Audio file pointer for the audio file
   -> const char Fname[]
      File name
   -> int Format
      Audio file data format code, evaluated as
        Format = Dformat + Ftype
      Dformat: data format
        FD_MULAW8  = 1,  mu-law 8-bit data
        FD_ALAW8   = 2,  A-law 8-bit data
        FD_UINT8   = 3,  offset-binary 8-bit integer data
        FD_INT8    = 4,  two's-complement 8-bit integer data
        FD_INT16   = 6,  two's-complement 16-bit integer data
        FD_FLOAT32 = 6,  32-bit float data
        FD_TEXT    = 7,  text data
      Ftype: file type
        FW_NH_EB     = 768, Headerless file (big-endian byte order)
        FW_NH_EL     = 1024, Headerless file (little-endian byte order)
        FW_NH_NATIVE = 1280, Headerless file (native byte order)
        FW_NH_SWAP   = 1536, Headerless file (swapped byte order)
   -> long int Nchan
      Number of channels
   -> double Sfreq
      Sampling frequency
   -> FILE *fpout
      File pointer for printing audio file information.  If fpout is not NULL,
      information about the audio file is printed on the stream selected by
      fpout.

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/10/29 17:26:07 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFsetNHpar.c 1.10 1996/10/29 AFsp-V2R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/AFpar.h>

AFILE *
AFsetNHpar (Fname, Format, Nchan, Sfreq, fpout)

     const char Fname[];
     int Format;
     long int Nchan;
     double Sfreq;
     FILE *fpout;

{
  FILE *fp;
  AFILE *AFp;
  int Dformat, Ftype, Swapb;

/* Decode the format into the data format and the file type */
  Dformat = Format % FW_FTYPE_MOD;
  Ftype = Format - Dformat;

/* Set up the encoding parameters */
  switch (Dformat) {
  case FD_MULAW8:
  case FD_ALAW8:
  case FD_UINT8:
  case FD_INT8:
  case FD_INT16:
  case FD_FLOAT32:
  case FD_TEXT:
    break;
  default:
    UThalt ("AFsetNHpar: Unsupported data encoding");
    break;
  }

/* Set up the swap parameter */
  switch (Ftype) {
  case FW_NH_EB:
    Swapb = DS_EB;
    break;
  case FW_NH_EL:
    Swapb = DS_EL;
    break;
  case FW_NH_NATIVE:
    Swapb = DS_NATIVE;
    break;
  case FW_NH_SWAP:
    Swapb = DS_SWAP;
    break;
  default:
    UThalt ("AFsetNHpar: Invalid file type");
    break;
  }

/* Open the file for writing */
  if (Dformat == FD_TEXT)
    fp = fopen (Fname, "w");	/* Open as a text file */
  else
    fp = fopen (Fname, "wb");	/* Open as a binary file */
  if (fp == NULL)
    UTerror ("AFsetNHpar: Cannot open file \"%s\"", Fname);

/* Set the parameters for file access */
  AFp = AFsetAFp (fp, FO_WO, FT_NH, Dformat, Swapb, 1.0, Nchan,
		  0L, 0L, 0L);

/* Print the header information */
  AFprintAFh (AFp, Fname, "", Sfreq, fpout);

  return AFp;
}
