/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AOsetDFormat (int Fformat, const int Dformat[], int Nf)

Purpose:
  Determine a compatible data format

Description:
  This routine finds a data format that is compatible with a number of input
  data formats and an output file type.  First, a data format with the same or
  greater precision as that for the input data formats is determined.  Second,
  based on this data format, the output data format is chosen to be compatible
  with the output file type.  For use in the error message, the program name
  should be set using the routine UTsetProg.

Parameters:
  <-  int AOsetDFormat
      File type and data format.  The file type is the same as that for
      Fformat.  If the data format component of Fformat is specified, that
      data format is used.  Otherwise, the data format is determined by the
      input data formats.
   -> int Fformat
      File type and data format
   -> const int Dformat[]
      Input data formats (Nf values)
   -> int Nf
      Number of input data formats (can be zero)

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/10/29 16:38:16 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AOsetDFormat.c 1.10 1996/10/29 AFsp-V2R2 $";

#include <libtsp.h>			/* defines AFILE, used by AFpar.h */
#include <libtsp/AFpar.h>
#include "AO.h"

#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

static const int *
AO_Allow p_((int Ftype));

/* Resulting precision for mixed data formats */
/* 0 - undefined, 1 - 16-bit, 2 - float */
static const int Prec[NFD] = { 0, 1, 1, 1, 1, 1, 2, 1};

/* Canonical data formats for each of the precisions */
static const int DfPrec[3] = {FD_UNDEF, FD_INT16, FD_FLOAT32};

int
AOsetDFormat (Fformat, Dformat, Nf)

     int Fformat;
     const int Dformat[];
     int Nf;

{
  int Ftype, i, Df, Dfr;
  const int *Allow;
  char *Prog;

/* Program name for error messages */
  Prog = (UTgetProg ())[0] == '\0' ? "AOsetDFormat" : UTgetProg ();

/* Find the file type and data format from Fformat */
  Dfr = Fformat % FW_FTYPE_MOD;
  Ftype = Fformat - Dfr;

/*
   Dfr       Ftype
  defined   defined    Use the combination specified by the user, but give an
                       error if the combination is unsupported.
  defined   undefined  If the data format is text, use a noheader file type,
                       otherwise use the default file type.  If the combination
                       of default file type and data format is unsupported, an
                       error message is printed (file type must be specified).
  undefined defined    Data type from input, modified by allowable output data
                       types for the given file type.
  undefined undefined  Data type from input, default output file type 
*/

  if (Dfr == FD_UNDEF) {

/* Data type not defined */
/* Use the default file type, if Ftype is undefined */
    if (Ftype == FW_UNDEF)
      Ftype = FW_DEFAULT;

    /* Choose the output data type based on the input data types
       Data promotion rules
       - convert to allowable data formats for the output file type
       - for mixed input formats, find the resulting precision
       - find the canonical data format for that precision
    */
    Allow = AO_Allow (Ftype);
    if (Nf <= 0)
      Dfr = Allow[FD_DEFAULT];
    else
      Dfr = Allow[Dformat[0]];
    for (i = 1; i < Nf; ++i) {
      Df = Allow[Dformat[i]];
      if (Df != Dfr)
	Dfr = DfPrec[MAXV (Prec[Df], Prec[Dfr])];
    }
  }

  else if (Ftype == FW_UNDEF) {

/* Data type defined, Ftype not defined */
    if (Dfr == FD_TEXT)
      Ftype = FW_NH_NATIVE;
    else
      Ftype = FW_DEFAULT;
    Allow = AO_Allow (Ftype);
    if (Dfr != Allow[Dfr])
      UThalt ("%s: Output file type must be specified", Prog);
  }

  else {

/* Data type defined, Ftype defined */

    /* Check the selected data type */
    Allow = AO_Allow (Ftype);
    if (Dfr != Allow[Dfr])
      UThalt ("%s: Invalid data format for selected output file type",
	      Prog);
  }

  return (Dfr + Ftype);
}

static const int *
AO_Allow (Ftype)

     int Ftype;

{
  const int *Allow;

/* Conversion tables to allowable data formats for different file types */
  static const int Allow_AFSP[NFD] = {
    FD_UNDEF, FD_MULAW8, FD_ALAW8, FD_INT8, FD_INT8, FD_INT16, FD_FLOAT32,
    FD_INT16
  };
  static const int Allow_WAVE[NFD] = {
    FD_UNDEF, FD_MULAW8, FD_ALAW8, FD_UINT8, FD_UINT8, FD_INT16, FD_INT16,
    FD_INT16
  };
  static const int Allow_AIFF_C[NFD] = {
    FD_UNDEF, FD_MULAW8, FD_ALAW8, FD_INT8, FD_INT8, FD_INT16, FD_INT16,
    FD_INT16
  };
  static const int Allow_NH[NFD] = {
    FD_UNDEF, FD_MULAW8, FD_ALAW8, FD_UINT8, FD_INT8, FD_INT16, FD_FLOAT32,
    FD_TEXT
  };

  switch (Ftype) {
  case FW_AFSP:
    Allow = Allow_AFSP;
    break;
  case FW_WAVE:
    Allow = Allow_WAVE;
    break;
  case FW_AIFF_C:
    Allow = Allow_AIFF_C;
    break;
  case FW_NH_EB:
  case FW_NH_EL:
  case FW_NH_NATIVE:
  case FW_NH_SWAP:
    Allow = Allow_NH;
    break;
  default:
    Allow = NULL;
    break;
  }

  return Allow;
}
