/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Routine:
  void CCoptions (int argc, const char *argv[], int *Fformat, float *SfreqO,
                  const char *NHparms[MAXIFILE], const char **Hinfo,
		  float *Gain, const char *Fname[MAXFILE], int *Nfiles)

Purpose:
  Decode options for ConcatAudio

Description:
  This routine decodes options for ConcatAudio.

Parameters:
   -> int argc
      Number of command line arguments
   -> const char *argv[]
      Array of pointers to argument strings
  <-  int *Fformat
      Output file format code
  <-  float *SfreqO
      Output file sampling frequency
  <-  const char *NHparms[MAXIFILE]
      Parameters for headerless input files, default NULL
  <-  const char **Hinfo
      Header information string, default NULL
  <-  float *Gain
      Gain value
  <-  const char *Fname[MAXFILE]
      File names
  <-  int *Nfiles
      Number of input file names

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.10 $  $Date: 1996/10/29 16:17:27 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CCoptions.c 1.10 1996/10/29 AFsp-V2R2 $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFpar.h>
#include "ConcatAudio.h"
#include "AO.h"

#ifndef EXIT_SUCCESS
#  define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define ERRSTOP(text,par)	UThalt ("%s: %s: \"%s\"", PROGRAM, text, par)
#define NELEM(array)	((sizeof array) / (sizeof array[0]))

/* Option tables and usage message */
#define LOPT	(NELEM (OptTable) / 2)
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-s#", "--sr*ate=",
  "-D#", "--d*ata_format=",
  "-F#", "--f*ile_type=",
  "-g#", "--g*ain=",
  "-P#", "--p*arameters=",
  "-I#", "--i*nfo=",
  "-h",  "--h*elp",
  "-v",  "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] AFileI1 AFileI2 ... AFileO\n\
Options:\n\
  -s SFREQ, --srate=SFREQ     Sampling frequency for the output file.\n\
  -D DFORMAT, --data_format=DFORMAT  Data format for the output file,\n\
                              \"mu-law\", \"A-law\", \"unsigned8\", \"integer8\",\n\
                              \"integer16\", \"float\", or \"text\".\n\
  -F FTYPE, --file_type=FTYPE  Output file type, \"AFsp\", \"WAVE\", \"AIFF-C\",\n\
                              \"noheader\", or \"noheader_swap\".\n\
  -g GAIN, --gain=GAIN        Gain factor.\n\
  -P PARMS, --parameters=PARMS  Parameters for headerless input files,\n\
                              \"Format,Start,Sfreq,Swapb,Nchan,ScaleF\".\n\
  -I INFO, --info=INFO        Header information string.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
CCoptions (argc, argv, Fformat, SfreqO, NHparms, Hinfo, Gain, Fname, Nfiles)

     int argc;
     const char *argv[];
     int *Fformat;
     float *SfreqO;
     const char *NHparms[MAXIFILE];
     const char **Hinfo;
     float *Gain;
     const char *Fname[MAXFILE];
     int *Nfiles;

{
  int Index;
  const char *OptArg;
  const char **optt;

  int nF, n, nn;
  float Sfreq;
  const char *NHp;
  float gain;
  int Dformat, Ftype;

/* Defaults */
  Sfreq = 0.0;
  NHp = NULL;
  gain = 1.0;
  Dformat = FD_UNDEF;
  Ftype = FW_UNDEF;
  *Hinfo = NULL;

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;

/* Decode options */
  Index = 1;
  optt = OptTable;
  while (Index < argc) {
    n = UTgetOption (&Index, argc, argv, optt, &OptArg);
    nn = ((n + 3) / 2) - 1;	/* n = -2 ==> nn = -1 */
    switch (nn) {
    case 0:
      /* Filename argument */
      ++nF;
      if (nF <= MAXFILE)
 	Fname[nF-1] = OptArg;
      else
	UThalt ("%s: Too many filenames specified", PROGRAM);
      if (nF < MAXIFILE)
	NHparms[nF-1] = NHp;
      break;
    case 1:
      /* Sampling rate */
      if (STdec1float (OptArg, &Sfreq) || Sfreq <= 0.0)
	ERRSTOP ("Invalid sampling frequency", OptArg);
      break;
    case 2:
      /* Data format */
      Dformat = AOdecDFormat (OptArg);
      break;
    case 3:
      /* File types */
      Ftype = AOdecFType (OptArg);
      break;
    case 4:
      /* Gain */
      if (STdec1float (OptArg, &gain))
	ERRSTOP ("Invalid gain value", OptArg);
      break;
    case 5:
      /* Headerless input parameters */
      NHp = OptArg;
      break;
    case 6:
      /* Header information string */
      *Hinfo = OptArg;
      break;
    case LOPT-2:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case LOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case LOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UThalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (nF < 2)
    UThalt ("%s: Too few files specified", PROGRAM);

/* Set return values */
  *Fformat = Dformat + Ftype;
  *SfreqO = Sfreq;
  *Gain = gain;
  *Nfiles = nF;

  return;
}
