/*------------- Telecommunications & Signal Processing Lab --------------
                          McGill University
Routine:
  void CAcomp (AFILE *AFpA, AFILE *AFpB, int Nsseg, int delayL, int delayU,
	       int *delayM, const struct Stats_F *StatsA,
	       const struct Stats_F *StatsB, struct Stats_T *StatsT)

Purpose:
  Gather correlation statistics for two audio files over a range of delays

Description:
  This routine gathers statistics for an audio files over a range of delays.
  The delay values runs from delayL to delayU.  For each delay the gain
  optimized SNR is calculated.  This value is printed for each delay value if
  delayL < delayU.  The cross-file statistics for the delay which maximizes the
  SNR are returned.

Parameters:
   -> AFILE *AFpA
      Audio file pointer for file A
   -> AFILE *AFpB
      Audio file pointer for file B
   -> long int Nsseg
      Segment length in samples for segmental SNR computations
   -> int delayL
      Start delay value.  Normally delayL <= delayU.  If this condition is not
      satisfied, no statistics are calculated.
   -> int delayU
      End delay value
  <-  int *delayM
      Delay of file B relative to file A which maximizes the gain adjusted SNR.
   -> const struct Stats_F *StatsA
      Structure containing the file statistics for file A
   -> const struct Stats_F *StatsB
      Structure containing the file statistics for file B
  <-  struct Stats_T *StatsT
      Structure containing the cross-file statistics corresponding to the delay
      value delayM

Author / revision:
  P. Kabal  Copyright (C) 1996
  $Revision: 1.11 $  $Date: 1996/06/01 02:41:10 $

-----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CAcomp.c 1.11 1996/06/01 AFsp-V2R2 $";

#include <math.h>		/* log10 */
#include <stdio.h>
#include "CompAudio.h"

#define ABSV(x)		(((x) < 0) ? -(x) : (x))
#define DB(x)		(10.0 * log10 (x))

void
CAcomp (AFpA, AFpB, Nsseg, delayL, delayU, delayM, StatsA, StatsB, StatsT)

     AFILE *AFpA;
     AFILE *AFpB;
     long int Nsseg;
     int delayL;
     int delayU;
     int *delayM;
     const struct Stats_F *StatsA;
     const struct Stats_F *StatsB;
     struct Stats_T *StatsT;

{
  struct Stats_T Statst;
  int delay;
  double SNR, SNRG, SNRGmax, SF, SSNR;

/* Loop over the delays */
  SNRGmax = -DBL_MAX;
  for (delay = delayL; delay <= delayU; ++delay) {

    /* Cross product terms */
    CAcorr (AFpA, AFpB, delay, Nsseg, &Statst);

    if (delayL != delayU) {

      /* Calculate the SNR values */
      CASNR (StatsA, StatsB, &Statst, &SNR, &SNRG, &SF, &SSNR);

      /* Print the match statistics */
      if (SNRG != DBL_MAX) {
	printf (" Delay: %3d,  SNR = %-9.5g dB", delay, DB (SNRG));
	printf ("  (Gain factor for File B = %.5g)\n", SF);
      }
      else
	printf (" Delay: %3d,  File A = %.5g * File B\n", delay, SF);
   }

    /* Save the best match statistics - if best match occurs at several
       delays, use the delay with the smallest absolute value */
    if (SNRG > SNRGmax) {
      SNRGmax = SNRG;
      *StatsT = Statst;
      *delayM = delay;
    }
    else if (SNRG == SNRGmax && ABSV (delay) <= ABSV (*delayM)) {
      *StatsT = Statst;
      *delayM = delay;
    }
  }

  return;
}
