/* -*- c++ -*-
 *
 *  Generalized MIDI parser. Loosely based on BeOS classes.
 *  Copyright (c) by Thomas Hudson <thudson@cygnus.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "MidiParser.h"
#include <stdlib.h>
#include <iostream.h>

MidiParser::MidiParser()
  : _running_status(0), _sysex_size(512) 
{ 
  _sysex_buf = (uchar*) malloc(_sysex_size); 
}

void MidiParser::parseMidi()
{
  uchar next = readByte();
  if (isStatus(next))
    {
      // If we receive a system realtime, we process immediately and
      // then continue with out normally scheduled programming
      if (isSystemRealTime(next))
	SpraySystemRealTime(next,NOW);

      // Set the sysex flag to start saving data bytes.
      else if (isSystemExclusive(next))
	_in_sysex = true;

      // Any other channelized status byte
      else
	{
	  // Receiving any status byte other than SysRealTime terminates
	  // a System Exclusive message
	  if (_in_sysex)
	    SpraySystemExclusive(_sysex_buf,_data_count);
	  
	  _running_status = next & 0xf0; // Strip off channel
	  _channel = next & 0x0f; // Save channel here
	  _data_count = 0; // Reset data count
	}
    }
  else // data byte
    {
      if (_in_sysex)
	{
	  if (_sysex_size == _data_count)
	    {
	      _sysex_size *= 2;
	      _sysex_buf = (uchar*)realloc(_sysex_buf,_sysex_size);
	    }
	  _sysex_buf[_data_count++] = next;
	}
      else
	{
	  // Add data byte to buffer and check counts for each message type
	  _data_buf[_data_count++] = next;
	  
	  switch (_running_status)
	    {
	    case NOTE_ON:
	      if (_data_count == 2)
		{
		  SprayNoteOn(_channel,_data_buf[0],_data_buf[1]);
		  _data_count = 0;
		}
	      break;
	      
	    case NOTE_OFF:
	      if (_data_count == 2)
		{
		  SprayNoteOff(_channel,_data_buf[0],_data_buf[1]);
		  _data_count = 0;
		}
	      break;
	      
	    case KEY_PRESSURE:
	      if (_data_count == 2)
		{
		  SprayKeyPressure(_channel,_data_buf[0],_data_buf[1]);
		  _data_count = 0;
		}
	      break;
	      
	    case CONTROL_CHANGE:
	      if (_data_count == 2)
		{
		  SprayControlChange(_channel,_data_buf[0],_data_buf[1]);
		  _data_count = 0;
		}
	      break;
	      
	    case PROGRAM_CHANGE:
	      if (_data_count == 1)
		{
		  SprayProgramChange(_channel,_data_buf[0]);
		  _data_count = 0;
		}
	      break;
	      
	    case CHANNEL_PRESSURE:
	      if (_data_count == 1)
		{
		  SprayChannelPressure(_channel,_data_buf[0]);
		  _data_count = 0;
		}
	      break;
	      
	    case PITCH_BEND:
	      if (_data_count == 2)
		{
		  SprayPitchBend(_channel,_data_buf[0],_data_buf[1]);
		  _data_count = 0;
		}
	      break;
	      
	    case SYSTEM_COMMON:	
	      if (_data_count  == 2)
		{
		  SpraySystemCommon(_running_status & _channel, _data_buf[0],_data_buf[2]);
		  _data_count = 0;
		  _running_status = 0;
		}
	      break;

	    default: // shouldn't get here
	      cout << "Status = " << _running_status; << endl;
	      for (int i = 0; i < _data_count; ++i)
		cout << " data[" << i << "] = " << _data_buf[i] << endl;
		  
	      break;
	    }
	}
    }
}
