/* Author - Arun Jagota 1/8/90 

		COPY RIGHT agreement
This software is given free of charge for research/academic purposes. 
Any changes may be made to the software for research purposes. 
It is recommended that publications stemming from using it cite
[1], [2] or both.

This software may be distributed anywhere for academic/research 
purposes as long as the terms outlined here are adhered to. This text 
must accompany the distribution. If there is any public installation
of this s/w, "Arun Jagota" should be listed as the author in the
'man' pages, if any.

Before any commercial use of this s/w (direct or indirect), Arun Jagota
must be consulted.  (jagota@cs.buffalo.edu)

[1] A new Hopfield-style network for content-addressable memories 
- Arun Jagota

[2] Knowledge Representation in a multi-layered hopfield network
- Arun Jagota, Oleg Jakubowicz, (IJCNN-89, Wash DC)

*/
/* Invocation

	mlhn <network> [{-tat -t -tOFF -ns -tP}] [-c] 

	-tat : Train and Test (Update weights, find_energy_minimum
			       and grow layers)
	-t   : Test only (Don't update weights for clamped units.
		Also don't create new units).
	-c   : Assume all units in input are clamped to 1 unless
	       <unit-name>*0
	-tOFF: Stop after switching units off only during testing
	-ns  : Don't automatically compute set size and use it for 
	       training. Eg { c a }. Train only { c a }, not { c a 2 }
	-top : Output only top layer output (>= 2 layers)
	-wta : Top layer is winner-take-all (>= 2 layers)
	-tP  : While testing, interpret 'R_i' as the a priori 
	       probabilities instead of internal resistance. This is 
	       done in the energy descent algorithm.

*/

/* Variable naming conventions *************************
IMPORTANT Global
----------------
    p     : 1-d array for unit values at all layers.
    w     : 1-d array for weights at all layers.
    wup   : 1-d array for each unit (contains upwards connections)
    pl    : starting index of layer 'l' units in 'p', 'p_clamped', 
	    'p_name_tbl', 'neti'
    wl	  : starting index of layer 'l' units in 'w'
    l     : layer no, 0 for the first layer
    lambda: Learning rate
    lambda_R: Learning rate of Threshold, R[x] for each unit 'x'.

Naming Conventions For Temporary variables and Parameters
---------------------------------------------------------
    pli   : index of 'ith' unit in layer 'l' in 'p'
    pli_c : index of 'ith' 'p_clamped' unit in layer 'l'
    p_name: unit name
    pi    : Offset of a unit from the start of it's layer 'l' (ie 'i')
    pi_c  : index of 'ith' 'p_clamped' unit from the start of it's 
	    layer 'l'
    wli   : starting index of 'ith' row in layer 'l' in 'w'
    wlij  : index of weight for pair of units (pli, plj) in 'w'

OTHER GLOBAL
------------
    p_name_tbl : 1-d array for pointers to unit name entries at all 
		 layers. The first max_size_units of p_name_tbl for 
		 each layer are assigned to the size units to keep 
		 a 1-1 correspondence between the index of a symbol's 
		 name and it's unit number.
    R          : 1-d array for unit thresholds at all layers
    r          : 1-d array for unit threshold transients at all layers
    neti       : 1-d array for unit net input values at all layers
    max_p      : array of 'max' units in each layer. Configuration
		 parameter max_p => there are (max_p-1) units in that
		 layer. Some of these units (first few) are hidden size 
		 units.
		 
    curr_p     : array of 'current' units in each layer, start = 1.
		 This includes hidden size units.
    max_size_units: array of 'size' units in each layer. Configuration
		 parameter. Out of the max_p-1 units, the 1st max_size_units
		 will be used to represent the sizes 1..max_size_units.
    curr_layers: Number of layers, start = 1 
    curr_clamped:array of 'currently clamped' units, start = 1
			       ************************/

/* This also does memory management of unit values, names and weights */
/* A 1-d array is used for all units (ie p). 
    first unit   max offset
    ----------   ----------
	0        - max_p[1]-1 : units in layer p1
	max_p[1] - max_p[2]-1 : units in layer p2
	etc....
   
   A similar breakup is used for the arrays 'p_name_tbl',
   'p_clamped' and 'neti'.

   The weights are distributed as follows.

   first weight	    max offset
   ------------     ----------
	0	   - (max_p[1])^2-1 : for layer p1 units
	max_p[1]^2 - (max_p[2])^2-1 : for layer p2 units
	etc ....
*/

#include<stdio.h>
#include<string.h>
#include "std.h"
#include "utils.h"

/* Total number of units available - For all layers*/
#define MAX_P 5000  

/* Total number of weights available - For all layers
   Must be >= max_p[1]^2 + max_p[2]^2 + max_p[3]^2 ...
*/
#define MAX_W 25000000 
#define MIN_VIGILANCE -200000000
#define POS_WGT 1

/* IF SCALE and lambda(l) are > 0, d_wlij is >= 0 */
/* So can use SHIFTS for Multiplication */

/* Macro for the 'quick' learning rule */

#ifdef QUICK
#define DELTA_WIJ(l,w_wlij,d_wlij,trials,pli,plj) \
{ \
d_wlij = (lambda[l]*(SCALE - abs(w_wlij)) >> SCALE_POWER); \
d_wlij = (d_wlij*p[pli]*p[plj]) >> (2*HIGH_POWER); \
} \

#else

/* Macro for a sigmoid learning rule */
#define DELTA_WIJ(l,w_wlij,d_wlij,trials,pli_val,plj_val) \
{ \
   trials = ((SCALE - w_wlij) << SCALE_POWER) / (SCALE + w_wlij); \
   d_wlij = ( (lambda[l] * trials) << (SCALE_POWER+1)) / \
	    ( (SCALE + trials) * (SCALE + trials)); \
   d_wlij = (d_wlij*pli_val*plj_val) >> (2*HIGH_POWER); \
} \

#endif

/* Macro for weight update rule */
/* Uses global variables 'w', 'p', 'lambda', 'vigilance' */
/* pli,plj MUST be clamped */


#ifdef UNLEARN

/* Macro for the non-Hebbian learning rule. It allows for fast 
removal of spurious minima but looses discriminatory capability
between stored memories (minima) */


#define UPDATE_WIJ(l,pli,plj,wlij,wlji,w_wlij,d_wlij,trials,pli_val,plj_val) \
{ \
  w_wlij = w[wlij]; \
  if (((p[pli] == CLAMPED_HIGH) && (p[plj] == CLAMPED_LOW)) || \
      ((p[pli] == CLAMPED_LOW)  && (p[plj] == CLAMPED_HIGH))) \
    { \
     if (w_wlij >= max_size_units[l])  w[wlij] = w[wlji] = -w_wlij/max_size_units[l]; \
     else {if (w_wlij >= 0) w[wlij] = w[wlji] = -1;} \
    } \
  else \
    { \
     if (w_wlij <= -max_size_units[l])  w[wlij] = w[wlji] = -w_wlij/max_size_units[l]; \
     else {if (w_wlij <= 0) w[wlij] = w[wlji] = 1;} \
    } \
} \

#else

/* Regular Hebbian but non-linear Update rule */

#define UPDATE_WIJ(l,pli,plj,wlij,wlji,w_wlij,d_wlij,trials,pli_val,plj_val) \
{ \
  w_wlij = w[wlij]; \
  if (((p[pli] == CLAMPED_HIGH) && (p[plj] == CLAMPED_LOW)) || \
      ((p[pli] == CLAMPED_LOW)  && (p[plj] == CLAMPED_HIGH))) \
    { \
     if (w_wlij == vigilance[l]) \
       w[wlij] = w[wlji] = -((lambda[l]*pli_val*plj_val) >> (2*HIGH_POWER)); \
     else { \
       DELTA_WIJ(l,w_wlij,d_wlij,trials,pli_val,plj_val) \
       if (w_wlij > 0) w_wlij--; \
       if ((w_wlij - (int) d_wlij) < 0) w[wlij] = w[wlji] = 0; \
       else w[wlij] = w[wlji] = w_wlij - d_wlij; \
     } \
    } \
  else \
    { \
     if (w_wlij == vigilance[l]) \
       w[wlij] = w[wlji] = (lambda[l]*pli_val*plj_val) >> (2*HIGH_POWER); \
     else { \
       DELTA_WIJ(l,w_wlij,d_wlij,trials,pli_val,plj_val) \
       if (w_wlij < 0) w_wlij++; \
       w[wlij] = w[wlji] = w_wlij + d_wlij; \
     } \
    } \
} \

#endif

/* Threshold learning */
#ifdef QUICK

#define DELTA_RI(l,R_rli,d_rli,trials,pli) \
{ \
d_rli = (lambda_R[l]*(THRESH_SCALE - R_rli) >> THRESH_SCALE_POWER); \
d_rli = (d_rli*p[pli]) >> HIGH_POWER; \
} \

#else

#define DELTA_RI(l,R_rli,d_rli,trials,pli) \
{ \
   trials = ((THRESH_SCALE - R_rli) << THRESH_SCALE_POWER) / (THRESH_SCALE + R_rli); \
   d_rli  = ( (lambda_R[l] * trials) << (THRESH_SCALE_POWER+1)) / \
	    ( (THRESH_SCALE + trials) * (THRESH_SCALE + trials)); \
   d_rli  = (d_rli*p[pli]) >> HIGH_POWER; \
} \

#endif

#define UPDATE_RI(l,pli,R_rli,d_rli,trials) \
{ \
  R_rli = R[pli]; \
  DELTA_RI(l,R_rli,d_rli,trials,pli) \
  if (p[pli] == CLAMPED_HIGH) \
     R[pli] = R_rli + d_rli; \
  else { \
  if (d_rli == 0) d_rli = 1; \
  if ((R_rli - (int) d_rli) < 0) R[pli] = 0; \
  else R[pli] = R_rli - d_rli; \
  } \
} \


/* Macro for computing 'pl' */
#define COMPUTE_PL(i,l) \
  { \
   pl = 0; \
   for (i = 0; i <  l; i++) \
    pl += max_p[i]; \
  }  \

/* Macro for computing 'wl' */
#define COMPUTE_WL(i,l) \
  { \
   wl = 0; \
   for (i = 0; i <  l; i++) \
    wl += max_p[i]*max_p[i]; \
  }  \



/* Represent real weights in interval -1.0
to +1.0 but multiplied by a factor of SCALE
*/
int *w;   

int wl;

/* Feedforward weights - Directed upwards */
struct wup_entry {
  int index, value;
  struct wup_entry *next;
};

struct wup_entry **wup;

/* number of incoming upwards connections (for each unit) */
int num_up[MAX_P];

/* pointers to Unit name entries for all layers */

/* Last entry in each layer terminated by the value 'NULL' */
P_NAME_ENTRY **p_name_tbl; 
P_NAME_ENTRY **hashtab[MAX_LAYERS]; 
int hashsize[MAX_LAYERS];

/* Units for all layers */
int *p;            

/* Thresholds for all units */
int *R;
/* Transient Threshold for all units */
int *r;
/* Global threshold */
int Rg[MAX_LAYERS];

/* Global variables, l and pl */
int l;
int pl;

/* Keeps track of the current number of units in each layer */
int curr_p[MAX_LAYERS],curr_layers;   

/* The indices of all units (per layer) that are clamped */
/* The value stored is the unit index in THAT LAYER NOT the starting
   index of that layer + the offset. p_clampedval stores the value of
   clamped units.
*/
int p_clamped[MAX_P], p_clamped_low[MAX_P],p_clampedval[MAX_P];  
/* Keeps track of how many units are currently clamped in each layer */
int curr_clamped[MAX_LAYERS],curr_clamped_low[MAX_P];

/* Command line flags - Default mode for unit names -  '*1' */
boolean all_clamped_high;
boolean test_only,test_mode = TEST_ONLY, train_and_test, compress_weights, no_size,top_layer_wta;

/* The net input to all units (per layer) - To find the last entry
   in each layer find the index of the last entry in the array 'p'
*/
int *neti;


/* The indices of all units (per layer) that are not LOW */
/* Used for propagation by dynamic programming     */
/* Linked list because entries must be dynamically removable 
   when they become LOW
*/
struct pl_not_low {
   int index;  /* Stores the index (from the layer's START) of a particular active unit */
   struct pl_not_low *next;
};

/* And now an array of the max number of layers */
struct pl_not_low *p_not_low[MAX_LAYERS];


FILE *cfg_fp,*w_fp,*p_fp,*p_names_fp,*R_fp, *wup_fp;
char file[WORD_SIZE];

/* Configuration parameters - to be read in that order (per line)*/
/* Vigilance should typically be -10 representing -1.0 */
int max_p[MAX_LAYERS],lambda[MAX_LAYERS],lambda_R[MAX_LAYERS],vigilance[MAX_LAYERS], max_size_units[MAX_LAYERS];

read_cfg_file(fp)
FILE *fp;
/* Format of input (all integers)
	max_p lambda lambda_R vigilance max_size_units

	^ Above line for each layer

*/
{unsigned num_units,num_weights;
  char *calloc();
  int i;

  num_units = 0;
  num_weights= 0;
  curr_layers = 0;
  while ((curr_layers <= MAX_LAYERS) && 
	 (fscanf(fp,"%d%d%d%d%d",&max_p[curr_layers],&lambda[curr_layers],
	  &lambda_R[curr_layers],&vigilance[curr_layers],&max_size_units[curr_layers]) != EOF)) {
	 num_units += max_p[curr_layers];
	 num_weights += max_p[curr_layers]*max_p[curr_layers];
	 if (vigilance[curr_layers] < MIN_VIGILANCE) {
    	    fprintf(stderr,"Vigilance should be > %d. Change .cfg file\n",MIN_VIGILANCE);
    	    exit(0);
	 }
	 curr_layers++;
  }

  if (top_layer_wta && (curr_layers == 1)) {
    fprintf(stderr,"1st layer cannot be 'wta' \n");
    exit(0);
  }

  if (top_layer_wta) num_weights -= max_p[curr_layers-1]*max_p[curr_layers-1];
  if (curr_layers > MAX_LAYERS) fprintf(stderr,"curr_layers too high \n");
  if (num_units > MAX_P) {
    fprintf(stderr,"number of units available is %d. Reduce number in .cfg file\n",MAX_P);
    exit(0);
  }

  if (num_weights > MAX_W) {
    fprintf(stderr,"number of weights available is %d. Change .cfg file\n",MAX_W);
    exit(0);
  }

  /* Now allocate the space for all the arrays */
  p = (int *) calloc(num_units,sizeof(int));
  r = (int *) calloc(num_units,sizeof(int));
  R = (int *) calloc(num_units,sizeof(int));
  neti = (int *) calloc(num_units,sizeof(int));
  w = (int *) calloc(num_weights,sizeof(int));
  p_name_tbl = (P_NAME_ENTRY **) calloc(num_units,sizeof(P_NAME_ENTRY *));
  wup = (struct wup_entry **) calloc(num_units,sizeof(struct wup_entry *));
  for (i = 0; i < curr_layers; i++) {
    hashsize[i] = max_p[i]/2;
    hashtab[i] = (P_NAME_ENTRY **) calloc(hashsize[i],sizeof(P_NAME_ENTRY *));
  }
}

restore_all_weights(argv)
/* Uses 'curr_p' to figure out how many weights to restore */
char *argv[];
{int l,wl,l_max;
 char str[3];

  l_max = (top_layer_wta) ? (curr_layers-1) : curr_layers;
  wl = 0;
  for (l = 0; l < l_max; l++) {
    strcpy(file,argv[1]);
    itoa(l+1,str);
    strcat(file,".w");
    strcat(file,str);
    w_fp = fopen(file,"r");
    if (w_fp != NULL) 
       /* restore_weights(w_fp,&w[wl],max_p[l],curr_p[l]); */
       restore_rl_weights(w_fp,&w[wl],max_p[l],curr_p[l],vigilance[l],
			  MIN_VIGILANCE); 
    else
       if (curr_p[l] > max_size_units[l]) {
	  fprintf(stderr,"Can't find weights file\n");
	  exit(0);
	}

    /*Compute starting position for weights in layer l+1 */
    wl += max_p[l]*max_p[l];
  }
}

save_all_weights(argv)
/* Uses curr_p to figure out how many weights to save */
char *argv[];
{int l,wl,l_max;
 char str[3];

  l_max = (top_layer_wta) ? (curr_layers-1) : curr_layers;
  wl = 0;
  for (l = 0; l < l_max; l++) {
    strcpy(file,argv[1]);
    itoa(l+1,str);
    strcat(file,".w");
    strcat(file,str);
    w_fp = fopen(file,"w");
    /* save_weights(w_fp,&w[wl],max_p[l],curr_p[l]); */
    save_rl_weights(w_fp,&w[wl],max_p[l],curr_p[l],vigilance[l],MIN_VIGILANCE); 
    /*Compute starting position for weights in layer l+1 */
    wl += max_p[l]*max_p[l];
  }
}

restore_all_p_names(argv)
char *argv[];
{int l,pl,wl,size_unit,temp;
 char str[3],sze[5];

 pl = 0;  /* Start position in p_name_tbl */
 wl = 0;  /* Start position of weights to be initialised */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".pn");
    itoa(l+1,str);
    strcat(file,str);
    p_names_fp = fopen(file,"r");
    if (p_names_fp != NULL) restore_p_names(p_names_fp,&p_name_tbl[pl],&curr_p[l],max_p[l],l);
    else /* Initialise symbols for the first max_size_units */
      { for (size_unit = 1; size_unit <= max_size_units[l]; size_unit++) {
	  itoa(size_unit,str);
	  strcpy(sze,"##");
	  strcat(sze,str);
	  find_or_create_unit(&p_name_tbl[pl],sze,CREATE_ONLY,&curr_p[l],&temp,max_p[l],l);
	  initialise_weights(&w[wl],(size_unit-1),vigilance[l],max_p[l],
	  curr_p[l]);
	}
      }
    pl += max_p[l];
    wl += max_p[l]*max_p[l];
 }
}

save_all_p_names(argv)
char *argv[];
{int l,pl;
 char str[3];

 pl = 0;  /* Start position in p_name_tbl */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".pn");
    itoa(l+1,str);
    strcat(file,str);
    p_names_fp = fopen(file,"w");
    save_p_names(p_names_fp,&p_name_tbl[pl]);  
    pl += max_p[l];
 }
}

restore_all_R(argv)
char *argv[];
{int l,pl;
 char str[3];

 pl = 0;  /* Start position in 'R' */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".R");
    itoa(l+1,str);
    strcat(file,str);
    R_fp = fopen(file,"r");
    if (R_fp != NULL) restore_R(R_fp,&R[pl],curr_p[l]);
    pl += max_p[l];
 }
}

save_all_R(argv)
char *argv[];
{int l,pl;
 char str[3];

 pl = 0;  /* Start position in 'R' */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".R");
    itoa(l+1,str);
    strcat(file,str);
    R_fp = fopen(file,"w");
    save_R(R_fp,&R[pl],curr_p[l]);  
    pl += max_p[l];
 }
}

restore_all_wup(argv)
char *argv[];
{int l,pl;
 char str[3];

 pl = 0;  /* Start position in 'R' */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".wup");
    itoa(l+1,str);
    strcat(file,str);
    wup_fp = fopen(file,"r");
    if (wup_fp != NULL) restore_wup(wup_fp,&wup[pl],curr_p[l],
				    &num_up[pl + max_p[l]]);
    pl += max_p[l];
 }
}

save_all_wup(argv)
char *argv[];
{int l,pl;
 char str[3];

 pl = 0;  /* Start position in 'R' */
 for (l = 0; l < curr_layers; l++) {
    strcpy(file,argv[1]);
    strcat(file,".wup");
    itoa(l+1,str);
    strcat(file,str);
    wup_fp = fopen(file,"w");
    save_wup(wup_fp,&wup[pl],curr_p[l]);  
    pl += max_p[l];
 }
}

int strip_weight(p_name)
char p_name[];
/* This function checks if the string in p_name begins with an integer 
   number between [1-99] followed by '-'. If not, it returns HIGH, 
   otherwise, it returns (weight*HIGH)/100 and strips p_name of its weight.
Syntax of weight, eg
70-A1 stands for symbol A1 having weight 70.
*/
{int weight; 
 int i,j,ln,pi_val; 

   if ((p_name[1] != '-') && ( p_name[2] != '-')) return HIGH;

   weight = atoi(p_name);
   if (weight == 0) return HIGH;
   i = 0;
   while ((p_name[i] >= '0') && (p_name[i] <= '9'))
     i++;

   i++; /* Skip the '-' */
   ln = strlen(&p_name[i]);
   for (j = 0; j <= ln; j++) {
      p_name[j] = p_name[i];
      i++;
   }
   pi_val = (weight << HIGH_POWER)/100;
   if (pi_val == 0) pi_val = 1;
   return (pi_val);
}

read_set_input(lp,plp,argv,curly)
/* Global variables 
	curr_layers, max_p, p, p_clamped, p_clamped_not_low, p_clampedval,
	p_not_low, curr_clamped, curr_clamped_low, test_only
*/

/* This function reads the unit names within '{' ... '}', 

a) determines the layer (say *lp) from the FIRST unit name in the set, 

b) determines the starting index (*plp) from layer *lp.

c) fills a table of clamped units (p_clamped[*plp]))

d) fills the unit activations (p[*plp]) [LOW-HIGH]

e) creates a list of all units not LOW (p_not_low[*lp]) 

*/

char *argv[]; /* IN */
int *lp,*plp; /* OUT */
boolean curly; /* If curly is FALSE, set very large positive weights
		  between units in the set and NO negative weights to
		  other units */

{int l,pl,pi_c,pi_cl,pi,pi_val; 
 struct pl_not_low *new;
/* 'pi_c' is offset in p_clamped array from 'pl'
   'pi'  is offset in 'p' array
   'pi_val' is value [LOW-HIGH] of unit from input.
*/
 char ch, p_name[WORD_SIZE];
 boolean new_created,find_or_what;

      ch = (curly) ? '}' : ']';

      scanf("%s",p_name);
      pi_val = strip_weight(p_name);
      if (test_only) find_or_what = FIND_ONLY;
      else find_or_what = FIND_OR_CREATE;

      /* First find the layer in which the first word is a unit name */
      /* If it is not in the first curr_layers -1 layers, then it must
	 be in the last layer */
      l = 0;
      pl = 0;
      while ((l < (curr_layers-1)) && 
	     ((pi = find_or_create_unit(&p_name_tbl[pl],p_name,FIND_ONLY,
	                                 &curr_p[0],&new_created,max_p[l],l)
	      )
	       == NOT_FOUND)
	    )
      {
        l++;
	pl += max_p[l];
      }
      if (l < (curr_layers-1)) *lp = l;  /* FOUND */
      else {
	*lp = curr_layers-1;
	pi = find_or_create_unit(&p_name_tbl[pl],p_name,find_or_what,
	      &curr_p[*lp],&new_created,max_p[*lp],l);
      }

      /* 'pl' is the starting index of layer '*lp' */
      *plp = pl;
      
      /* If '*lp' is top layer and top_layer_wta then error */
      if (top_layer_wta && ((*lp + 1) == curr_layers)) {
	fprintf(stderr,"Input not allowed to top layer - wta \n");
	exit(0);
      }

      pi_c = curr_clamped[*lp];
      pi_cl= curr_clamped_low[*lp];
      do { 
	if (pi == NOT_FOUND) {
        scanf("%s",p_name);
	pi_val = strip_weight(p_name);
	continue;
	}

	if (new_created && curly) 
	    initialise_weights(&w[wl],pi,vigilance[l],max_p[l],curr_p[l]);

	if (clamped_low(p_name)) {
	   p[pl+pi] = CLAMPED_LOW;
	   p_clamped_low[pl+pi_cl] = pi;
	   pi_cl++;
	   p_clamped[pl+pi_c] = pi;
	   p_clampedval[pl+pi_c] = pi_val;
 	   pi_c++;
	}
	else {
          /* Check if clamped_high*/
          if (all_clamped_high || clamped_high(p_name) ) {
	     p[pl+pi] = CLAMPED_HIGH;
	     p_clamped[pl+pi_c] = pi;
	     p_clampedval[pl+pi_c] = pi_val;
 	     pi_c++;
	  }
	  else
	     p[pl+pi] = pi_val;

	  /* Also insert this unit's index into the p_not_low list */
	  new = (struct pl_not_low *) malloc (sizeof (*new));
	  if (new == NULL) 
	    {fprintf(stderr,"Out of memory for pl_not_low\n");
	     save_all_weights(argv);
	     save_all_p_names(argv);
	     save_all_R(argv);
	     save_all_wup(argv);
	     exit(0);
	    }
	  new->next      = p_not_low[*lp];
	  new->index     = pi;
	  p_not_low[*lp] = new;

	}


        scanf("%s",p_name);
	pi_val = strip_weight(p_name);
      } while ((p_name[0] != ch) && 
	       ((pi = find_or_create_unit(&p_name_tbl[pl],p_name,
		      find_or_what,&curr_p[*lp],&new_created,max_p[*lp],l)
	       ) || TRUE
	       )
	      );

      /* Compute curr_clamped */
      curr_clamped[*lp] = pi_c;
      curr_clamped_low[*lp] = pi_cl;

}

delete_list(p_not_low)
struct pl_not_low *p_not_low;
{struct pl_not_low *this, *prev;

   if (p_not_low == NULL) return;

   prev = this = p_not_low;
   while (this->next != NULL) {
     prev = this;
     this = this->next;
     free(prev);
   }
   free(this);
}

reset_neti(l,pl)
int l,pl;
/* Resets only net input to each unit at layer 'l' */
{int i,pl1;
  /* Reset neti */
  pl1 = pl + curr_p[l];
  for (i = pl; i < pl1; i++) neti[i] = 0; 

}

reset_p(l,pl)
int l,pl;
/* Resets activity at layer 'l' */
{int i,pl1;
  /* This will automatically also set p_clamped[l] to 0 */
  curr_clamped[l] = 0;
  curr_clamped_low[l] = 0;

  /* Deallocate p_not_low[l] */
  delete_list(p_not_low[l]); 
  p_not_low[l] = NULL;

  /* Reset neti, p */
  pl1 = pl + curr_p[l];
  for (i = pl; i < pl1; i++) p[i] = neti[i] = 0; 

}

boolean unit_is_clamped_high(l,pl,pi)
/* Check if pi is clamped high */
int l,pl,pi;
{int j;
  for (j = 0; j < curr_clamped[l]; j++)
    if (p_clamped[pl+j] == pi) return TRUE;
  return FALSE;
}

propagate_up(p_not_low,next_p_not_lowp,l,pl,pl_gm)
/* Global variables read/modified 
	p, neti, wup, num_up
	curr_p
*/
/* Don't propagate up to clamped units */
struct pl_not_low *p_not_low, **next_p_not_lowp;
int l,pl,pl_gm;
{struct pl_not_low *pnl;
 struct wup_entry *wupp;
 int pli, pli_gm;
 boolean clamped_hi;
 int num_active = 0;  /* Use this */
 int num_clamped;

/* Find how many units are active */
 for (pnl = p_not_low; pnl != NULL; pnl = pnl->next) num_active++;
 num_clamped = curr_clamped[l];

 /* Propagate activation values up as neti */
 for (pnl = p_not_low; pnl != NULL; pnl = pnl->next) {
   pli = pl + (pnl->index);
   clamped_hi = unit_is_clamped_high(l,pl,pnl->index);
   for (wupp = wup[pli]; wupp != NULL; wupp = wupp->next) {
     pli_gm = pl_gm + wupp->index;
     /* Since the network has settled to a local minimum, the unit
	values are binary { 0,1 }
     */
     if (clamped_hi) 
	/* Use Rule 2) in KR paper */
	neti[pli_gm] += ((wupp->value)*num_clamped)/((num_clamped+1)*(num_clamped+num_up[pli_gm]));
	/* neti gets a value in the range of SCALE */
     else 
	/* Use Rule 2) in KR paper */
       neti[pli_gm] += (wupp->value)/((num_clamped+1)*(num_active+num_up[pli_gm]));
	/* neti gets a value in the range of SCALE */
   }
 }

 /* Compute activation values for all NON-SIZE units at layer 'l+1' */
 for (pli_gm = (pl_gm+max_size_units[l+1]); pli_gm < pl_gm + curr_p[l+1]; pli_gm++) {
      if ((p[pli_gm] != CLAMPED_HIGH)) {
	/* Use Rule 2) in KR paper */
	p[pli_gm] = (neti[pli_gm] << HIGH_POWER) >> (SCALE_POWER); 
	/* Now add 'pli_gm' to p_not_low list IF p[pli_gm] > 0*/

	if (p[pli_gm] == 0) continue;
	pnl = (struct pl_not_low *) malloc(sizeof(*pnl));
	pnl->next = *next_p_not_lowp;
	pnl->index= pli_gm - pl_gm;
	*next_p_not_lowp = pnl;
      }
     /* Now set neti to 0 */
     neti[pli_gm] = 0;
  }
}


grow_wup(pl,npi, p_not_low,p_clamped_low)
/* This 'grows' upwards connections from layer 'l' to 'npi'
based on which units are active (not low). Currently, upwards
connections have only one value - SCALE.
It also sets num_up[npl + npi]
*/

int pl, npi;
struct pl_not_low *p_not_low;
int p_clamped_low[];
{struct wup_entry *wupp, *wupnew;
 struct pl_not_low *pnl;
 int pli,npl;

 npl = pl + max_p[l];
 for (pnl = p_not_low; pnl != NULL; pnl = pnl->next) {
    pli = (pnl->index) + pl;
    for (wupp = wup[pli]; wupp != NULL; wupp = wupp->next)
      if (wupp->index == npi) break;

    if (wupp == NULL) {
      wupnew = (struct wup_entry *) malloc (sizeof (*wupnew));
      wupnew->index = npi;
      wupnew->next  = wup[pli];
      wupnew->value = SCALE;
      wup[pli]      = wupnew;
      num_up[npl + npi]++;
    }
 }
}

clamp_high(nl)
int nl;
/* Clamp all units in p_not_low to high */

{struct pl_not_low *pnl;
 int npl,pi_c;

 npl = pl + max_p[l];
 pi_c = curr_clamped[nl];

 for (pnl = p_not_low[nl]; pnl != NULL; pnl = pnl->next) {
   p_clamped[npl + pi_c] = pnl->index;
   p[npl + (pnl->index)] = CLAMPED_HIGH;
   pi_c++;
 }

 curr_clamped[nl] = pi_c;
}

adjust_weights(l,pl,wl,num_iter,curly,test_only,no_size)
int l,pl,wl,num_iter;  
/* If curly is FALSE, set a LARGE Positive weight */
/* If test_only is TRUE, don't UPDATE weights */
boolean curly, test_only,no_size;  
/* Global variables used.
   p_clamped, p, w, max_p, lambda, vigilance, p_clampedval
*/
/* First updates weights (Including self-weights). Then updates threshold.
*/
{int pi,pj,pli,plj,pi_c,pj_c, num_clamped, wli, wlij, wlji,pli_val,plj_val;
 int w_wlij;  /* Local variables in UPDATE_WIJ */
 int R_rli ;  /* Local variable in UPDATE_RI */
 unsigned int d_wlij,trials; /* Local variables in UPDATE_WIJ */
			     /* Unsigned because they store << values */
 unsigned int d_rli; /* Local variables in UPDATE_RI */
			     /* Unsigned because they store << values */
 int n_i;


   num_clamped = curr_clamped[l];

   
   /* Update weights between all pairs of clamped units */
   for (pi_c = 0; pi_c < (num_clamped-1); pi_c++) {
     pi  = p_clamped[pl+pi_c];
     pli_val = p_clampedval[pl+pi_c];
     pli = pl + pi;
     wli = wl + pi*max_p[l];
     for (pj_c = pi_c+1; pj_c < num_clamped; pj_c++) {
       pj   = p_clamped[pl+pj_c];
       plj_val = p_clampedval[pl+pj_c];
       plj  = pl + pj;
       wlij = wli + pj;
       wlji = wl + pj*max_p[l] + pi; 

       if (!curly) {
	  w[wlij] = w[wlji] = -vigilance[l]*POS_WGT;
	  continue;
       }
       
       if (! test_only) {
       for (n_i = 0; n_i < num_iter; n_i++)
         UPDATE_WIJ(l,pli,plj,wlij,wlji,w_wlij,d_wlij,trials,pli_val,plj_val)
       }

     }

   }
    
   /* New addition - Update weights between all clamped units and
      the size unit that represents the set size, which is  'num_clamped'
       if it exists, ie num_clamped <= max_size_units and no_size is FALSE*/

   if (!no_size && (num_clamped <= max_size_units[l])) {
     /* the size unit */
     pj = num_clamped-1;
     plj  = pl + pj;
     for (pi_c = 0; pi_c < num_clamped; pi_c++) {
       pi  = p_clamped[pl+pi_c];
       pli_val = p_clampedval[pl+pi_c];
       pli = pl + pi;
       wli = wl + pi*max_p[l];
       wlij = wli + pj;
       wlji = wl + pj*max_p[l] + pi; 
       if (! test_only) {
       for (n_i = 0; n_i < num_iter; n_i++)
         UPDATE_WIJ(l,pli,plj,wlij,wlji,w_wlij,d_wlij,trials,pli_val,pli_val)
       }
     }
   }

   /* Now update self-weights and threshold*/
   for (pi_c = 0; pi_c < num_clamped; pi_c++)  {
     pi  = p_clamped[pl + pi_c];
     pli_val = p_clampedval[pl+pi_c];
     pli = pl + pi;
     wli = wl + pi*max_p[l];
     wlij= wli + pi;

     if (! test_only) {
       for (n_i = 0; n_i < num_iter; n_i++) {
         UPDATE_WIJ(l,pli,pli,wlij,wlij,w_wlij,d_wlij,trials,pli_val,pli_val)
	 UPDATE_RI(l,pli,R_rli,d_rli,trials)
       }
     }

   }

   /* Update self-weights and threshold of size units also */
   /*  Let R_i of size units stay to zero.
   if (!no_size && (num_clamped <= max_size_units[l])) {
    pi = num_clamped-1;
    pli= pl + pi;
    wli = wl + pi*max_p[l];
    wlij= wli + pi;
    if (! test_only) {
     for (n_i = 0; n_i < num_iter; n_i++) {
       UPDATE_WIJ(l,pli,pli,wlij,wlij,w_wlij,d_wlij,trials)
       UPDATE_RI(l,pli,R_rli,d_rli,trials)
     }
    }
    }
   */
}

process_set_input(test_also,argv,num_iter,no_size)
boolean test_also,no_size;
char *argv[];
int  num_iter; /* Number of iterations of input - Only for learning */

{int i; /* Local variable to COMPUTE_WL */
 boolean curly;

    curly = TRUE;
    read_set_input(&l,&pl,argv,curly);
    COMPUTE_WL(i,l)
    adjust_weights(l,pl,wl,num_iter,curly,test_only,no_size);
    if (test_also) {
      find_energy_minimum(&p[pl],&R[pl],&r[pl],&w[wl],curr_p[l],max_p[l],&p_not_low[l]
			,&p_clamped[pl],&p_clamped_low[pl],&curr_clamped[l],
			curr_clamped_low[l],
			&neti[pl],FALSE,test_mode);
    }
}

check_growup(gm_create,gm)
boolean gm_create;
char gm[];
{int npi,npl,nwl,new_created;

   /* Now check if upwards connections have to and can be grown */
   if (gm_create && ((l+1) < curr_layers)) {
	npl = pl + max_p[l];
	nwl = wl + max_p[l]*max_p[l];
	npi = find_or_create_unit(&p_name_tbl[npl],gm,FIND_OR_CREATE,
		  &curr_p[l+1],&new_created,max_p[l+1],l);
	grow_wup(pl,npi,p_not_low[l],&p_clamped_low[pl]);
	if (new_created) 
	    initialise_weights(&w[nwl],npi,vigilance[l+1],max_p[l+1],curr_p[l+1]);
   }
}

main(argc,argv)
int argc;
char *argv[];
{int npl, nwl,num_iter;   
 char word[WORD_SIZE], gm[WORD_SIZE];
 boolean gm_create,curly,top_only;
 int i; /* Local variable for COMPUTE_WL */

if (argc == 1) printf("Usage : %s <network> [{-tat -t}] [-c] \n",argv[0]);
/* Configuration file */
strcpy(file,argv[1]);
strcat(file,".cfg");
cfg_fp = fopen(file,"r");
if (cfg_fp == NULL) {
    fprintf(stderr,"No config file : %s \n",file);
    exit(0);
    }

if (argc == 4) {
    train_and_test = (strcmp(argv[2],"-tat") == MATCH);
    test_only      = (strcmp(argv[2],"-t") == MATCH);
    if (strcmp(argv[2],"-tOFF") == MATCH) test_mode = TEST_OFF_ONLY;
    if (strcmp(argv[2],"-tP") == MATCH) test_mode = TEST_PROB;
    if ((test_mode == TEST_OFF_ONLY) || (test_mode == TEST_PROB)) test_only = TRUE;
    no_size      = (strcmp(argv[2],"-ns") == MATCH);
    all_clamped_high = (strcmp(argv[3],"-c") == MATCH);
    top_only = (strcmp(argv[3],"-top") == MATCH);
    top_layer_wta = (strcmp(argv[3],"-wta") == MATCH);
}

if (argc == 3) {
    train_and_test = (strcmp(argv[2],"-tat") == MATCH);
    test_only      = (strcmp(argv[2],"-t") == MATCH);
    if (strcmp(argv[2],"-tOFF") == MATCH) test_mode = TEST_OFF_ONLY;
    if (strcmp(argv[2],"-tP") == MATCH) test_mode = TEST_PROB;
    if ((test_mode == TEST_OFF_ONLY) || (test_mode == TEST_PROB)) test_only = TRUE;
    no_size      = (strcmp(argv[2],"-ns") == MATCH);
    all_clamped_high = (strcmp(argv[2],"-c") == MATCH);
    top_only = (strcmp(argv[2],"-top") == MATCH);
    top_layer_wta = (strcmp(argv[2],"-wta") == MATCH);
}

if (argc < 3) top_layer_wta = top_only = test_only = all_clamped_high = train_and_test = no_size = FALSE;

read_cfg_file(cfg_fp);

restore_all_p_names(argv);
/* restore_all_p_names sets the curr_p counter so should be done before
   restore_all_R, restore_all_weights */

restore_all_R(argv);
restore_all_wup(argv);
restore_all_weights(argv);

/* restore_all_p_states();*/

/* input scan, weight updating and energy minimization */

while (scanf("%s",word) != EOF) {

  if (isi(word)) {/* Number of iterations before input is given */
    num_iter = atoi(word);
    scanf("%s",word);
  }
  else num_iter = 1;
   
  if (word[0] == '[') {
    curly = FALSE;
    read_set_input(&l,&pl,argv,curly);
    COMPUTE_WL(i,l)
    adjust_weights(l,pl,wl,num_iter,curly,test_only,no_size);
    reset_p(l,pl);
    continue;
  }

  gm_create = FALSE;
  if (strcmp(word,"->") == MATCH) {
     scanf("%s%s",gm,word);
     gm_create = TRUE;
  }

  if (word[0] == '{') {
      curly = TRUE;
      process_set_input((train_and_test || test_only),argv,num_iter,no_size);
      if ((train_and_test || test_only) && 
	  (!((top_only || top_layer_wta)  && ((l+1) < curr_layers))))
         display_p(&p_name_tbl[pl],p_not_low[l],max_size_units[l],test_mode);
      check_growup(gm_create,gm);
      npl = pl + max_p[l];
      if ((l+1) < curr_layers) {
        propagate_up(p_not_low[l],&p_not_low[l+1],l,pl,npl);
	nwl = wl + max_p[l]*max_p[l];
	if (top_layer_wta && ((l+2) == curr_layers))
		find_wta(&p[npl],curr_p[l+1],&p_not_low[l+1]);
	else
        	find_energy_minimum(&p[npl],&R[npl],&r[npl],&w[nwl],curr_p[l+1],
			max_p[l+1],&p_not_low[l+1] ,&p_clamped[npl],&p_clamped_low[npl],
			&curr_clamped[l+1],
			curr_clamped_low[l+1], &neti[npl], TRUE,test_mode);
        display_p(&p_name_tbl[npl],p_not_low[l+1],max_size_units[l+1],test_mode);
      }
      reset_p(l,pl);
      reset_p(l+1,npl);
      continue;
  }

  if (word[0] == '<') {
      scanf("%s",word);

      while (word[0] != '>') {
	process_set_input(TRUE,argv,1,FALSE);
	npl = pl + max_p[l];
        propagate_up(p_not_low[l],&p_not_low[l+1],l,pl,npl);

	nwl = wl + max_p[l]*max_p[l];
        find_energy_minimum(&p[npl],&R[npl],&r[npl],&w[nwl],curr_p[l+1],
			max_p[l+1],&p_not_low[l+1], &p_clamped[npl],&p_clamped_low[npl],
			&curr_clamped[l+1],
			curr_clamped_low[l+1], &neti[npl], TRUE,test_mode);
	
	clamp_high(l+1);
	reset_p(l,pl);
	reset_neti(l+1,npl); 
	scanf("%s",word);
      }  /* End while */

      /* Now adjust weights at 'l+1', display_p and then reset_p*/

      adjust_weights(l+1,npl,nwl,1,TRUE,FALSE,no_size);
      display_p(&p_name_tbl[npl],p_not_low[l+1],max_size_units[l+1],test_mode);
      reset_p(l+1,npl);

      continue;
  } /* End sequential input, ie '>' */

  if (word[0] == 'w') {int tl, twl, tpl,i; 
     /* Save l, pl, wl */
     tl = l; tpl = pl; twl = wl; 

     scanf("%d",&l);
     l--;
     COMPUTE_PL(i,l)
     COMPUTE_WL(i,l)
     display_weights(&p_name_tbl[pl],&w[wl],max_p[l],l);

     /* Restore l, pl, wl */
     l = tl; pl = tpl; wl = twl;
     continue;
   }
  if (word[0] == 'r') {int tl, twl, tpl,i; 
     /* Save l, pl, wl */
     tl = l; tpl = pl; twl = wl; 

     scanf("%d",&l);
     l--;
     COMPUTE_PL(i,l)
     COMPUTE_WL(i,l)
     display_R(&p_name_tbl[pl],&R[pl],max_p[l],l);

     /* Restore l, pl, wl */
     l = tl; pl = tpl; wl = twl;
     continue;
   }
}  /* End of ALL INPUT */

/* Now save the weights at all layers */
save_all_weights(argv);

/* Now save the unit names , thresholds at all layers */
save_all_p_names(argv);
save_all_R(argv);
save_all_wup(argv);
}
